<?php

namespace Widget\DiscountBundle\Controller\BackendAPI;


use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints\NotBlank;
use Widget\DiscountBundle\Model\DiscountGroup;
use Widget\DiscountBundle\Model\DiscountGroupPeer;
use Widget\DiscountBundle\Model\DiscountGroupQuery;
use Widget\DiscountBundle\Model\DiscountModuleQuery;
use Widget\DiscountBundle\Model\DiscountToGroup;
use Widget\DiscountBundle\Model\DiscountToGroupQuery;
use Widget\MemberBundle\Model\Group;

/**
 * @Route("/discountgroup")
 * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP')")
 */
class DiscountGroupController extends BaseBackendAPIController
{
    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.discountgroup.name.missing_field',
                    )),
                )                
            )),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP_WRITE')")
     */
    public function createAction(Request $request)
    {
        // 抽離出來做（同樣）動作 參數： 物件、content
        return $this->doProcessForm(new DiscountGroup(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP_READ')")
     */
    public function searchAction(Request $request)
    {
        return $this->doSearch($request->query->all(), DiscountGroupQuery::create()->distinct(), DiscountGroupPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP_READ')")
     */
    public function readAction(DiscountGroup $discountGroup)
    {
        return $this->createJsonSerializeResponse($discountGroup, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP_WRITE')")
     */
    public function updateAction(Request $request, DiscountGroup $discountGroup)
    {
        return $this->doProcessForm($discountGroup, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP_WRITE')")
     */
    public function deleteAction(DiscountGroup $discountGroup)
    {
        $this->deleteObject($discountGroup);
        return $this->createJsonResponse();
    }

    /**
     * 設置非互斥的關聯
     * @Route("/{id}/discounttogroup")
     * @Method({"POST"})
     * @param Request $request
     * @param DiscountGroup $discountGroup
     * @return array
     */
    public function relateAction(Request $request, DiscountGroup $discountGroup)
    {
        $requestParams = json_decode($request->getContent(), true);

        if (!isset($requestParams['modules']) || count($requestParams['modules']) == 0) {
            $emptyModule = DiscountModuleQuery::create()->findPks(array());
            $discountGroup->setDiscountModules($emptyModule)->save();
            return $this->createJsonSerializeResponse($discountGroup, array('list'));
        }

        // 清掉舊的非互斥關聯
        DiscountToGroupQuery::create()
            ->filterByGroupId($discountGroup->getId())
            ->filterByExclude(false)
            ->delete();
        // 寫入表單設定的非互斥關聯
        foreach ($requestParams['modules'] as $module){
            $relate = DiscountToGroupQuery::create()
                ->filterByDiscountModuleId($module['id'])
                ->filterByGroupId($discountGroup->getId())
                ->findOneOrCreate();
            $relate->setExclude(false)->save();
        }

        return $this->createJsonSerializeResponse($discountGroup, array('list'));
    }

    /**
     * 設置互斥的關聯
     * @Route("/{id}/discounttogroup/exclude")
     * @Method({"POST"})
     * @param Request $request
     * @param DiscountGroup $discountGroup
     * @return array
     */
    public function excludeRelateAction(Request $request, DiscountGroup $discountGroup)
    {
        $requestParams = json_decode($request->getContent(), true);

        if (!isset($requestParams['exclude_modules']) || count($requestParams['exclude_modules']) == 0) {
            $emptyModule = DiscountModuleQuery::create()->findPks(array());
            $discountGroup->setDiscountModules($emptyModule)->save();
            return $this->createJsonSerializeResponse($discountGroup, array('list'));
        }

        // 清掉舊的互斥關聯
        DiscountToGroupQuery::create()
            ->filterByGroupId($discountGroup->getId())
            ->filterByExclude(true)
            ->delete();
        // 寫入表單設定的互斥關聯
        foreach ($requestParams['exclude_modules'] as $module){
            $relate = DiscountToGroupQuery::create()
                ->filterByDiscountModuleId($module['id'])
                ->filterByGroupId($discountGroup->getId())
                ->findOneOrCreate();
            $relate->setExclude(true)->save();
        }

        return $this->createJsonSerializeResponse($discountGroup, array('list'));
    }

    /**
     * @Route("s/all")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTMODULE_WRITE')")
     */
    public function allAction(Request $request)
    {
        $discountGroup = DiscountGroupQuery::create()->find();
        return $this->createJsonSerializeResponse($discountGroup, array('list'));
    }

    /**
     * 設定 群組 & 優惠群組 關聯
     * @Route("/{id}/group")
     * @Method({"POST"})
     */
    public function relateToGroup(Request $request, Group $group)
    {
        $requestParams = json_decode($request->getContent(), true);
        if (!isset($requestParams['discount_groups']) || count($requestParams['discount_groups']) == 0) {
            $emptyDiscountGroup = DiscountGroupQuery::create()->findPks(array());
            $group->setDiscountGroups($emptyDiscountGroup)->save();
            return $this->createJsonSerializeResponse($group, array('list'));
        }

        $discountGroupIds = array();
        $discountGroupIds[] = $requestParams['discount_groups'];
        $discountGroups = DiscountGroupQuery::create()->findPks($discountGroupIds);
        $group->setDiscountGroups($discountGroups)->save();
        return $this->createJsonSerializeResponse($requestParams);
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNTGROUP_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $discountGroups = DiscountGroupQuery::create()->findPks($ids, $con);
        foreach ($discountGroups as $discountGroup) {
            $value = $discountGroup->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $discountGroup->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $discountGroup->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        DiscountGroupQuery::create()->findPks($ids, $con)->delete($con);
    }
}