<?php

namespace Widget\DiscountBundle\Service;

use JMS\DiExtraBundle\Annotation as DI;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Widget\Discount\TargetAmountBundle\Discount\TargetAmountDiscount;
use Widget\DiscountBundle\Event\DiscountFactoryEvent;
use Widget\DiscountBundle\Model\DiscountModule;
use Widget\DiscountBundle\Model\DiscountModuleQuery;
use Widget\DiscountBundle\Model\DiscountToGroup;
use Widget\DiscountBundle\Model\DiscountToGroupQuery;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\MemberBundle\Model\Group;
use Widget\MemberBundle\Model\GroupQuery;
use Widget\MemberBundle\Model\Member;
use Widget\MemberBundle\Model\MemberQuery;
use Widget\OrderBundle\Model\Order;

/**
 * @DI\Service("widget.discount_bundle.discount_builder")
 */
class DiscountBuilder
{

    /** @var DiscountInterface[] */
    protected $discounts = array();

    /** @var DiscountInterface[]  */
    protected $excludeDiscounts = array();

    protected $order;

    protected $conn;


    /** @var  EventDispatcherInterface */
    protected $eventDispatcher;
    
    protected $parameters;

    /**
     * @DI\InjectParams()
     */
    public function injectEventDispatcher(EventDispatcherInterface $eventDispatcher)
    {
        $this->eventDispatcher = $eventDispatcher;
    }

    public function init(Order $order, \PropelPDO $conn = null, $parameters = array())
    {
        $this->order = $order;
        $this->conn = $conn;
        $this->parameters = $parameters;
        if ($member = MemberQuery::create()->findPk($order->getMemberId(), $this->conn)){
            $this->prepareDiscountsRule($member);
        }        
    }

    public function prepareDiscountsRule(Member $member)
    {
        $this->prepareDiscounts($member);
        $this->prepareExcludeDiscounts($member);
    }
    
    protected function prepareDiscounts(Member $member)
    {
        $discountModules = $this->groupToDiscount($member->getGroups());
        /** @var DiscountModule $module */
        foreach ($discountModules as $module) {
            $module = $this->factory($module);
            if ( $module != null ) {
                $this->discounts[] = $module;
            }
        }
    }
    
    protected function prepareExcludeDiscounts(Member $member)
    {

    }

    /**
     * 從會員群組撈出所有的優惠條件
     * @param $memberGroups
     * @return DiscountModule[]
     */
    protected function groupToDiscount($memberGroups)
    {
        $discountModules = array();
        $discountModuleQuery = DiscountModuleQuery::create()
            ->useDiscountToGroupQuery()
                ->useDiscountGroupQuery()
                    ->useDiscountGroupToGroupQuery()
                        ->filterByGroup($memberGroups)
                    ->endUse()
                ->endUse()
            ->endUse()
            ->orderBySort(\Criteria::ASC)
            ->orderById(\Criteria::ASC)
            ->distinct()
            ->find();

        /** @var DiscountModule $discountModule */
        foreach ($discountModuleQuery as $discountModule) {
            $discountModules[$discountModule->getId()] = $discountModule;
        }

        return array_values($discountModules);
    }
    
    /**
     * @param $module
     * @return mixed
     */
    public function factory(DiscountModule $module)
    {
        $event = new DiscountFactoryEvent($module->toArray(\BasePeer::TYPE_FIELDNAME), $this->parameters);
        $this->eventDispatcher->dispatch(DiscountFactoryEvent::EVENT_NAME, $event);

        if ( $event->getObject() InstanceOf DiscountInterface ) {
            return $event->getObject();
        }
        return null;
    }
    
    /**
     * 加入優惠折扣（非互斥關係）
     * @param DiscountInterface $discount
     * @return $this
     */
    public function add(DiscountInterface $discount)
    {
        $this->discounts[] = $discount;
        return $this;
    }
    
    /**
     * 加入優惠折扣（互斥關係）
     * @param DiscountInterface $discount
     * @return $this
     */
    public function addExclude(DiscountInterface $discount)
    {
        $this->excludeDiscounts[] = $discount;
        return $this;
    }

    /**
     * 執行優惠折扣計算
     * @param Order $order
     * @param \PropelPDO|null $con
     */
    public function run(Order $order, \PropelPDO $con = null)
    {
        $this->runExcludeDiscounts($order, $con);
        $this->runDiscounts($order, $con);
    }

    /**
     * 計算有互斥的優惠折扣
     * @param Order $order
     * @param \PropelPDO|null $con
     */
    protected function runExcludeDiscounts(Order $order, \PropelPDO $con = null)
    {
        foreach ($this->excludeDiscounts as $discount) {
            $discount->process($order, $con);
            if ($discount->isProcessed()) {
                break;
            }
        }
    }

    /**
     * 計算非互斥的優惠折扣
     * @param Order $order
     * @param \PropelPDO|null $con
     */
    protected function runDiscounts(Order $order, \PropelPDO $con = null)
    {
        foreach ($this->discounts as $discount) {
            $discount->process($order, $con);
        }
    }
}