<?php

namespace Widget\Discount\TargetAmountBundle\Tests\Discount;


use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Widget\Discount\TargetAmountBundle\Discount\TargetAmountDiscount;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;

class TargetAmountDiscountTest extends BaseTestCase
{
    public function test_create_empty_config()
    {
        // arrange
        $config = array();
        
        // act
        $targetAmount = TargetAmountDiscount::create($config);
        
        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_empty_name_config()
    {
        // arrange
        $config = array(
            'config' =>
                array(
                    0 => array(
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);

        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_empty_amount_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'config' =>
                array(
                    0 => array(
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);

        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_empty_discount_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);

        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_empty_start_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => 'fake date',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);

        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_empty_stop_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'stop_at' => 'fake date',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);

        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_start_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => 'fake date',
            'stop_at' => new \DateTime('now'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);
        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_stop_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => new \DateTime('now'),
            'stop_at' => 'fake date',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);

        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_stop_not_in_time_range()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => new \DateTime('2099-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);
        // assert
        $this->assertNull($targetAmount);
    }

    public function test_create_ok()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );

        // act
        $targetAmount = TargetAmountDiscount::create($config);
        // assert
        $this->assertInstanceOf(DiscountInterface::class, $targetAmount);
    }

    public function test_process_ok()
    {
        // arrange
        $config = array(
            'name' => '滿千送百',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );
        $targetAmount = TargetAmountDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($targetAmount, 'config', $config);

        // act
        $targetAmount->process($order);

        // assert
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals('滿千送百', $discountOrderItem->getProductName());
        $this->assertEquals(-100, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    /**
     * 折到變負數 折扣最高為訂單總價
     */
    public function test_process_zero()
    {
        // arrange
        $config = array(
            'name' => '滿千送百',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 10,
                        'discount' => 100
                    )
                )
        );
        $targetAmount = TargetAmountDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($targetAmount, 'config', $config);

        // act
        $targetAmount->process($order);

        // assert
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals('滿千送百', $discountOrderItem->getProductName());
        $this->assertEquals(-10, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
        $this->assertEquals(0 ,$order->getAmountForPay());
    }

    public function test_process_not_enough()
    {
        // arrange
        $config = array(
            'name' => '滿千送百',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );
        $targetAmount = TargetAmountDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(800)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addOrderItem($orderItem);

        $this->setObjectAttribute($targetAmount, 'config', $config);

        // act
        $targetAmount->process($order);

        // assert
        $this->assertEquals(1, $order->countOrderItems());
    }

    // 折扣後不符合折扣條件 不進入優惠模組
    public function test_process_not_enough_had_discount()
    {
        // arrange
        $config = array(
            'name' => '滿千送百',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'discount' => 100
                    )
                )
        );
        $targetAmount = TargetAmountDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItemDiscount = new OrderItem();
        $orderItem
            ->setUnitPrice(1000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $orderItemDiscount
            ->setUnitPrice(-1)
            ->setQuantity(1)
            ->setProductName('discount')
            ->setProductId(null)
        ;        
        $order->addOrderItem($orderItem);
        $order->addOrderItem($orderItemDiscount);
        $order->syncAmount();

        $this->setObjectAttribute($targetAmount, 'config', $config);

        // act
        $targetAmount->process($order);

        // assert
        $this->assertEquals(2, $order->countOrderItems());
        $this->assertEquals(999, $order->getAmountForPay());
    }    
}