<?php

namespace Widget\Discount\ShipmentBundle\Tests\Discount;

use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Widget\Discount\ShipmentBundle\Discount\ShipmentDiscount;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;

class ShipmentDiscountTest extends BaseWebTestCase
{
    public function test_create_empty_config()
    {
        // arrange
        $config = array();

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_empty_name_config()
    {
        // arrange
        $config = array(
            'config' =>
                array(
                    0 => array(
                        'shipment' => 100
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_empty_amount_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'config' =>
                array(
                    0 => array(
                        'shipment' => 100
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_empty_discount_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_empty_start_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => 'fake date',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_empty_stop_config()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'stop_at' => 'fake date',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_start_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => 'fake date',
            'stop_at' => new \DateTime('now'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);
        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_stop_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => new \DateTime('now'),
            'stop_at' => 'fake date',
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);

        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_stop_not_in_time_range()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => new \DateTime('2099-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'shipment' => 100
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);
        // assert
        $this->assertNull($shipmentDiscount);
    }

    public function test_create_ok()
    {
        // arrange
        $config = array(
            'name' => 'Discount Name',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                        'shipment' => 100
                    )
                )
        );

        // act
        $shipmentDiscount = ShipmentDiscount::create($config);
        // assert
        $this->assertInstanceOf(DiscountInterface::class, $shipmentDiscount);
    }

    /**
     * 符合免運條件
     */
    public function test_process_ok()
    {
        // arrange
        $config = array(
            'name' => '滿千免運',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );
        $shipmentDiscount = ShipmentDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $shipmentItem = new OrderItem();
        $shipmentItem->setUnitPrice(100);
        $shipmentItem->setQuantity(1);
        $shipmentItem->setProductName('運費');

        $order->addProducts($orderItem);
        $order->addShipments($shipmentItem);

        $this->setObjectAttribute($shipmentDiscount, 'config', $config);

        // act
        $shipmentDiscount->process($order);

        // assert
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        
        $this->assertEquals('滿千免運', $discountOrderItem->getProductName());
        $this->assertEquals(-100, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    // 不符合免運條件
    public function test_process_not_enough()
    {
        // arrange
        $config = array(
            'name' => '滿千免運',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );
        $shipmentDiscount = ShipmentDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(800)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($shipmentDiscount, 'config', $config);

        // act
        $shipmentDiscount->process($order);

        // assert
        $this->assertCount(0, $order->getDiscounts());
    }

    // 折扣後不符合免運條件 不扣運費
    public function test_process_not_enough_had_discount()
    {
        // arrange
        $config = array(
            'name' => '滿千免運',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1000,
                    )
                )
        );
        $shipmentDiscount = ShipmentDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItemDiscount = new OrderItem();
        $orderItem
            ->setUnitPrice(1000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $orderItemDiscount
            ->setUnitPrice(-1)
            ->setQuantity(1)
            ->setProductName('discount')
            ->setProductId(null)
        ;
        $order->addProducts($orderItem);
        $order->addDiscounts($orderItemDiscount);

        $this->setObjectAttribute($shipmentDiscount, 'config', $config);

        // act
        $shipmentDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getDiscounts());
        $this->assertEquals(999, $order->getOrderAmountForPay());
    }
}