<?php

namespace Widget\Discount\ProductBundle\Tests\Discount;

use Backend\BaseBundle\Tests\Fixture\BaseWebTestCase;
use Widget\CategoryBundle\Model\Category;
use Widget\CategoryBundle\Model\CategoryQuery;
use Widget\Discount\ProductBundle\Discount\ProductDiscount;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;
use Widget\ProductBundle\Model\ProductQuery;

class ProductDiscountTest extends BaseWebTestCase
{
    public function test_process_not_repeat()
    {
        // arrange
        $products = ProductQuery::create()->orderByCreatedAt(\Criteria::DESC)->limit(2)->find();
        $product1 = $products[0];
        $product2 = $products[1];
        $config = array(
            'name' => '單品折扣',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'discount_name'   => '特定產品10%',
                        'discount_category' => $product1->getCategories(),
                        'discount' => 10,
                        'rule' => 'not_repeat',
                    ),
                    1 => array(
                        'discount_name'   => '產品折20%喔',
                        'discount_category' => $product2->getCategories(),
                        'discount' => 20,
                        'rule' => 'not_repeat',
                    ),
                    3 => array(
                        'discount_name'   => '不會被折到',
                        'discount_category' => $product1->getCategories(),
                        'discount' => 40,
                        'rule' => 'not_repeat',
                    ),
                ),
        );
        
        $productDiscount = ProductDiscount::create($config);

        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(1000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId($product1->getId())
        ;
        $orderItem2 = new OrderItem();
        $orderItem2
            ->setUnitPrice(2000)
            ->setQuantity(1)
            ->setProductName('test_product2')
            ->setProductId($product2->getId())
        ;

        $order->addProducts($orderItem);
        $order->addProducts($orderItem2);

        $this->setObjectAttribute($productDiscount, 'config', $config);

        // act
        $productDiscount->process($order);

        // assert
        $this->assertCount(2, $order->getDiscounts());
        $this->assertTrue($this->getObjectAttribute($productDiscount, 'processed'));
    }

    public function test_process_unique()
    {
        // arrange
        $product = ProductQuery::create()->orderByCreatedAt(\Criteria::DESC)->findOne();
        $config = array(
            'name' => '單品折扣',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'discount_name'   => '特定產品10%',
                        'discount_category' => $product->getCategories(),
                        'discount' => 10,
                        'rule' => 'unique',
                    ),
                    1 => array(
                        'discount_name'   => '產品折20%喔',
                        'discount_category' => $product->getCategories(),
                        'discount' => 20,
                        'rule' => 'unique',
                    ),
                ),
        );

        $productDiscount = ProductDiscount::create($config);

        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(1000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId($product->getId())
        ;

        $order->addProducts($orderItem);

        $this->setObjectAttribute($productDiscount, 'config', $config);

        // act
        $productDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getDiscounts());
        $this->assertTrue($this->getObjectAttribute($productDiscount, 'processed'));

    }

    public function test_checkDiscountCategory()
    {
        // assert
        $config = $this->getDefaultConfig();

        $productDiscount = ProductDiscount::create($config);

        $category = CategoryQuery::create()->filterByName('root')->find();

        $product = ProductQuery::create()->orderById(\Criteria::DESC)->findOne();
        foreach ($product->getCategories() as $productCategory) {
            $category = $productCategory;
            break;
        }
        $orderItem = new OrderItem();
        $orderItem->setUnitPrice(10000);
        $orderItem->setQuantity(1);
        $orderItem->setProductId($product->getId());

        // act
        $result  = $this->callObjectMethod($productDiscount, 'checkDiscountCategory', $category, $orderItem);

        // assert
        $this->assertTrue($result);
    }

    public function test_calcDiscountAmount()
    {
        // assert
        $config = $this->getDefaultConfig();
        $productDiscount = ProductDiscount::create($config);
        $discountParams = array(
            'itemQuantity' => 10,
            'unitPrice' => 7749,
            'discount' => 67,
        );
        $orderItem = new OrderItem();
        $orderItem->setQuantity($discountParams['itemQuantity']);
        $orderItem->setProductId(1);
        $orderItem->setUnitPrice($discountParams['unitPrice']);

        $discountConfig = array(
            'discount' => $discountParams['discount'],
        );

        // 按計算機的結果
        $expectDiscountAmount = 51918;

        // act
        $amount = $this->callObjectMethod($productDiscount, 'calcDiscountAmount', $orderItem, $discountConfig);

        // assert
        $this->assertEquals($expectDiscountAmount, $amount);
    }

    public function test_addOrderDiscount()
    {
        // assert
        $config = $this->getDefaultConfig();
        $productDiscount = ProductDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $discountAmount = 100;
        $config = array(
            'discount_name' => '就是有折扣',
        );
        // act
        $this->callObjectMethod($productDiscount, 'addOrderDiscount', $order, $orderItem, $discountAmount, $config);

        // assert
        /** @var OrderItem $discountItem */
        $discountItem = $order->getDiscounts()[0];
        $this->assertEquals(-$discountAmount, $discountItem->getUnitPrice());
        $this->assertEquals(1, $discountItem->getQuantity());

    }

    protected function getDefaultConfig()
    {
        return array(
            'name' => '單品折扣',
            'start_at' => new \DateTime('200-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'discount_name'   => '全部產品折20%喔',
                        'discount_category' => CategoryQuery::create()->filterByName('root')->find(),
                        'discount' => 10,
                        'rule' => 'not_repeat',
                    ),
                    1 => array(
                        'discount_name'   => '全部產品折20%喔',
                        'discount_category' => CategoryQuery::create()->filterByName('root')->find(),
                        'discount' => 20,
                        'rule' => 'not_repeat',
                    ),
                ),
        );
    }
}