<?php

namespace Widget\Discount\ProductBundle\Discount;


use Widget\CategoryBundle\Model\Category;
use Widget\DiscountBundle\Service\Discount\AbstractDiscount;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;
use Widget\ProductBundle\Model\ProductQuery;

class ProductDiscount extends AbstractDiscount
{
    protected $config;

    protected $orderProductProcessArray = array();

    public static function create($config)
    {
        if (!isset($config['config']) || !isset($config['config'][0])) {
            return null;
        }

        if (!isset($config['name'])) {
            return null;
        }

        if (!isset($config['start_at']) || !isset($config['stop_at'])) {
            return null;
        }

        if (!$config['start_at'] instanceof \DateTime || !$config['stop_at'] instanceof \DateTime) {
            return null;
        }

        if (time() < $config['start_at']->getTimeStamp() || time() > $config['stop_at']->getTimeStamp()) {
            return null;
        }

        return parent::create($config);
    }

    public function process(Order $order, \PropelPDO $con = null)
    {
        $discountConfig = $this->config['config'];

        // 建立 Array 儲存產品是否有折扣過
        $orderProductProcessedArray = array();
        /** @var OrderItem $item */
        foreach ($order->getProducts() as $item) {
            $orderProductProcessedArray[$item->getProductId()] = false;
        }

        $ruleProcessed = false;
        foreach ($discountConfig as $config) {
            /** @var Category $category */
            foreach ($config['discount_category'] as $category){
                /** @var OrderItem $orderItem */
                foreach ($order->getProducts() as $orderItem) {
                    // 商品已跑過折扣且設定不可重複折扣 下一輪
                    if ($orderProductProcessedArray[$orderItem->getProductId()]
                        && $config['rule'] == 'not_repeat'
                    ) {
                        continue;
                    }
                    // 商品不再折扣分類內 下一輪
                    if (!$this->checkDiscountCategory($category, $orderItem, $con)) {
                        continue;
                    }

                    $discountAmount = $this->calcDiscountAmount($orderItem, $config);
                    $this->addOrderDiscount($order, $orderItem, $discountAmount, $config);

                    // 設定本商品已折扣及本條件已跑過折扣
                    $orderProductProcessedArray[$orderItem->getProductId()] = true;
                    $ruleProcessed = true;
                }
            }
            // 設定只跑第一個符合的 而且跑過了 之後就不跑
            if ($config['rule'] == 'unique' && $ruleProcessed == true) {
                break;
            }
        }

        // 有 rule 處理過 模組已處理
        if ($ruleProcessed) {
            $this->processed = true;
        }
    }

    /**
     * 判斷產品分類是否屬於優惠分類
     * @param \PropelPDO $con
     * @param Category $category
     * @param OrderItem $orderItem
     * @return bool
     */
    protected function checkDiscountCategory($category, $orderItem, \PropelPDO $con = null)
    {
        $isInDiscountCategory = false;
        $product = ProductQuery::create()->findPk($orderItem->getProductId(), $con);

        /** @var Category $productCategory */
        foreach ($product->getCategories() as $productCategory) {
            /** 條件符合或設定根目錄就表示全折了 */
            if ($category->getId() == $productCategory->getId() || $category->getTreeLevel() == 0) {
                $isInDiscountCategory = true;
            }
        }

        return $isInDiscountCategory;
    }

    /**
     * 計算應該扣金額
     * @param OrderItem $orderItem
     * @param $config
     * @return int
     */
    protected function calcDiscountAmount(OrderItem $orderItem, $config)
    {
        $subTotal = $orderItem->getUnitPrice() * $orderItem->getQuantity();
        $discountPercentage = round($config['discount'] / 100, 2) ;
        return $discountAmount = round($subTotal * $discountPercentage);
    }


    /**
     * 訂單加入折扣
     * @param Order $order
     * @param OrderItem $orderItem
     * @param $discountAmount
     * @param $config
     */
    protected function addOrderDiscount(Order $order, OrderItem $orderItem, $discountAmount, $config)
    {
        $discountItem = new OrderItem();
        $discountItem->setProductName("{$orderItem->getProductName()} {$config['discount_name']}");
        $discountItem->setUnitPrice(-$discountAmount);
        $discountItem->setQuantity(1);
        $discountItem->setId(null);
        $order->addDiscounts($discountItem);
    }
}