<?php

namespace Widget\Discount\ProductBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Event\Controller\ConfigEvent;
use Backend\BaseBundle\Event\Controller\MenuEvent;
use Backend\BaseBundle\Event\Controller\TypeRolesEvent;
use Backend\BaseBundle\Form\Type;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Routing\Router;
use Symfony\Component\Validator\Constraints;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Widget\CategoryBundle\Model\Category;
use Widget\CategoryBundle\Model\CategoryQuery;
use Widget\DiscountBundle\Model\DiscountModule;

/**
 * @DI\Service()
 */
class ProductListener
{
    const CONTROLLER_NAME = 'WidgetDiscountProductBundle:Backend/Product';

    /** @var  Router */
    protected $router;

    /** @var  Session */
    protected $session;

    /**
     * 注入session服務
     * @InjectParams()
     */
    public function injectSession(Session $session)
    {
        $this->session = $session;
    }

    /**
     * 注入router服務
     * @InjectParams()
     */
    public function injectRouter(Router $router)
    {
        $this->router = $router;
    }

    /**
     * 設定選單
     * @DI\Observe(MenuEvent::EVENT_MENU)
     */
    public function onMenu(MenuEvent $event)
    {
        $user = $event->getUser();
        // 驗證是否有權限
        if(!$user->hasRoleOrSuperAdmin('ROLE_DISCOUNT_PRODUCT_READ')) {
            return;
        }

        $isActive = preg_match('/^widget_discount_product_backend_/i', $event->getRequestRouteName());
        $slug = $user->getSiteSlug();

        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.discount_group' => array(
                'class' => 'fa fa-money',
                'active' => $isActive,
                'target' => array(
                    'menu.discount_product' => array(
                        'class' => 'fa fa-money',
                        'active' => $isActive,
                        'target' => $this->router->generate('widget_discount_product_backend_product_index', array('slug' => $slug)),
                    ),
                ),
            ),
        ));
    }

    /**
     * 讓帳號管理介面加入權限設定
     * @DI\Observe(TypeRolesEvent::EVENT_TYPE_ROLES)
     */
    public function onTypeRoles(TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_DISCOUNT_PRODUCT', array(
            'ROLE_DISCOUNT_PRODUCT_READ',
            'ROLE_DISCOUNT_PRODUCT_WRITE',
        ));
    }

    /**
     * @DI\Observe(ConfigEvent::EVENT_CONFIG)
     */
    public function onConfig(ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->stopPropagation();
        $event->setConfig(array(
            'model' => DiscountModule::class,
            'route' => array(
                'index' => true,
                'sort' => true,
                'new' => true,
                'edit' => true,
                'delete' => true,
            ),
            'index' => array(
                '#' => 'Id',
                'index.discount_product.name' => 'Name',
                'index.discount_product.sort' => 'Sort',
                'index.discount_product.start_at' => 'StartAt',
                'index.discount_product.stop_at' => 'StopAt',
            ),
            'form' => array(
                array(
                    'Name',
                    BaseType\TextType::class,
                    array(
                        'label' => 'index.discount_product.name',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Config',
                    BaseType\CollectionType::class,
                    array(
                        'label' => 'index.discount_product.config.config',
                        'type' => new Type\BaseFormType(function(FormBuilderInterface $builder){
                            $builder
                                ->add('rule', BaseType\ChoiceType::class, array(
                                    'label'    => 'index.discount_product.config.rule',
                                    'multiple' => false,
                                    'expanded' => true,
                                    'choices'  => array(
                                        'not_repeat' => 'form.discount_product.rule.not_repeat',
                                        'unique' => 'form.discount_product.rule.unique',
                                        'all' => 'form.discount_product.rule.all'
                                    ),
                                ))
                                ->add('discount_name', BaseType\TextType::class, array(
                                    'label' => 'index.discount_product.config.discount_name',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                    ),
                                ))
                                ->add('discount_category', Type\RelationType::class, array(
                                    'label' => 'index.discount_product.config.discount_category',
                                    'class' => Category::class,
                                    'query' => CategoryQuery::create()
                                                ->useCategoryThreadQuery()
                                                    ->filterByThread('product')
                                                ->endUse(),
                                    'multiple' => true,
                                    'required' => false,
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                    ),
                                ))
                                ->add('discount', BaseType\IntegerType::class, array(
                                    'label' => 'index.discount_product.config.discount',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                    ),
                                ))
                            ;
                        }, 'Config'),
                        'allow_add' => true,
                        'allow_delete'  => true,
                    ),
                ),
                array(
                    'Sort',
                    BaseType\IntegerType::class,
                    array(
                        'label' => 'index.discount_product.sort',
                        'constraints' => array(
                            new Constraints\NotBlank(),
                            new Constraints\GreaterThan(array(
                                'value' => 0,
                            )),
                        ),
                    )
                ),
                array(
                    'ClassName',
                    BaseType\HiddenType::class,
                    array(
                        'label' => 'index.discount_product.start_at',
                    )
                ),
                array(
                    'StartAt',
                    BaseType\DateType::class,
                    array(
                        'label' => 'index.discount_product.start_at',
                        'required' => true,
                        'widget' => 'single_text',
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'StopAt',
                    BaseType\DateType::class,
                    array(
                        'label' => 'index.discount_product.stop_at',
                        'required' => true,
                        'widget' => 'single_text',
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'widget_discount_product_backend_product_index',
                    ),
                ),
            ),
            'security' => array(
                'new' => 'ROLE_DISCOUNT_PRODUCT_WRITE',
                'edit' => 'ROLE_DISCOUNT_PRODUCT_WRITE',
                'delete' => 'ROLE_DISCOUNT_PRODUCT_WRITE',
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
        ));
    }
}