<?php

namespace Widget\Discount\PercentageBundle\Tests\Discount;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Widget\Discount\PercentageBundle\Discount\PercentageDiscount;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;

class PercentageDiscountTest extends BaseTestCase
{
    public function test_create_empty_config()
    {
        // arrange
        $config = array();

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_empty_discount_config()
    {
        // arrange
        $config = array(
            'config' => array(
                0 => array()
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_empty_discount_amount_config()
    {
        // arrange
        $config = array(
            'config' => array(
                0 => array(
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_empty_discount_percentage_config()
    {
        // arrange
        $config = array(
            'config' => array(
                0 => array(
                    'amount' => 1,
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_empty_start_at()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'stop_at' => 'fake date',
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_empty_stop_at()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => 'fake date',
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_start_at_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => 'fake date',
            'stop_at' => new \DateTime('now'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_stop_at_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => new \DateTime('now'),
            'stop_at' => 'fake date',
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_stop_not_in_time_range()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => new \DateTime('2099-12-01'),
            'stop_at' => new \DateTime('2099-12-31'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertNull($percentage);
    }

    public function test_create_ok()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $percentage = PercentageDiscount::create($config);

        // assert
        $this->assertInstanceOf(DiscountInterface::class, $percentage);
    }

    public function test_process_ok_1()
    {
        // arrange
        $config = array(
            'name' => '老會員全產品七折',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1,
                        'percentage' => 30
                    )
                )
        );
        $percentageDiscount = PercentageDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($percentageDiscount, 'config', $config);

        // act
        $percentageDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(1, $order->getDiscounts());
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals($config['name'], $discountOrderItem->getProductName());
        $this->assertEquals(-3000, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    public function test_process_ok_2()
    {
        // arrange
        $config = array(
            'name' => 'VIP全產品六七折',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1,
                        'percentage' => 33
                    )
                )
        );
        $percentageDiscount = PercentageDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(7777)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($percentageDiscount, 'config', $config);

        // act
        $percentageDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(1, $order->getDiscounts());
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals($config['name'], $discountOrderItem->getProductName());
        $this->assertEquals(-2566, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    public function test_process_ok_3()
    {
        // arrange
        $config = array(
            'name' => 'MVP全產品 87% OFF',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 1,
                        'percentage' => 87
                    )
                )
        );
        $percentageDiscount = PercentageDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(77777)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($percentageDiscount, 'config', $config);

        // act
        $percentageDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(1, $order->getDiscounts());
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals($config['name'], $discountOrderItem->getProductName());
        $this->assertEquals(-67666, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    public function test_process_no_discount()
    {
        // arrange
        $config = array(
            'name' => 'MVP全產品 87% OFF',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' =>
                array(
                    0 => array(
                        'amount'   => 100,
                        'percentage' => 87
                    )
                )
        );
        $percentageDiscount = PercentageDiscount::create($config);
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($percentageDiscount, 'config', $config);

        // act
        $percentageDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(0, $order->getDiscounts());
    }
}