<?php

namespace Widget\Discount\PercentageBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Backend\BaseBundle\Form\Type\BaseFormType;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\Form\Extension\Core\Type\CollectionType;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints\GreaterThan;
use Symfony\Component\Validator\Constraints\LessThan;
use Symfony\Component\Validator\Constraints\NotBlank;
use Widget\Discount\PercentageBundle\Discount\PercentageDiscount;
use Widget\DiscountBundle\Model\DiscountModule;
use Widget\DiscountBundle\Model\DiscountModulePeer;
use Widget\DiscountBundle\Model\DiscountModuleQuery;

/**
 * @Route("/discount/percentage")
 * @Security("has_role_or_superadmin('ROLE_DISCOUNT_PERCENTAGE_READ')") 設定權限
 */
class PercentageController extends BaseBackendAPIController
{
    /**
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.discount_percentage.name.missing_field',
                    )),
                )
            )),
            new APIFormTypeItem('class_name'),
            (new APIFormTypeItem('sort'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.discount_percentage.sort.missing_field',
                    )),
                    new GreaterThan(array(
                        'value' => -1,
                        'message' => "error.discount_coupon.greater_than_negative",
                    ))                    
                )
            )),
            (new APIFormTypeItem('config'))->setFieldType(CollectionType::class)->setOptions(array(
                'type' => new BaseFormType(function(FormBuilderInterface $builder){
                    $builder
                        ->add('amount', IntegerType::class, array(
                            'label' => 'index.discount_percentage.config.amount',
                            'constraints' => [
                                new NotBlank(),
                                new GreaterThan([
                                    'value' => -1,
                                    'message' => 'error.discount_percentage.greater_than_negative'                                    
                                ]),
                            ],
                        ))
                        ->add('percentage', IntegerType::class, array(
                            'label' => 'index.discount_percentage.config.percentage',
                            'constraints' => array(
                                new NotBlank(),
                                new GreaterThan(array(
                                    'value' => -1,
                                    'message' => 'error.discount_percentage.greater_than_negative'
                                )),
                                new LessThan(array(
                                    'value' => 101,
                                    'message' => 'error.discount_percentage.less_than_100'                                    
                                ))
                            ),
                        ))
                    ;
                }, 'Config'),
                'allow_add' => true,
                'allow_delete'  => true,                
            )),
            (new APIFormTypeItem('start_at'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.discount_percentage.start_at.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('stop_at'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.discount_percentage.stop_at.missing_field',
                    )),
                )
            )),
        );
    }
    
    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNT_PERCENTAGE_WRITE')")
     */
    public function createAction(Request $request)
    {
        // 抽離出來做（同樣）動作 參數： 物件、content
        return $this->doProcessForm(new DiscountModule(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNT_PERCENTAGE_READ')")
     */
    public function searchAction(Request $request)
    {
        // query 列表 or 搜尋時 手動加入滿額折扣條件
        $defaultSearchArray = $request->query->get('search', array());
        $defaultSearchArray['class_name'] = PercentageDiscount::CLASS_NAME;
        $request->query->set('search', $defaultSearchArray);        
        return $this->doSearch($request->query->all(), DiscountModuleQuery::create()->distinct(), DiscountModulePeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNT_PERCENTAGE_READ')")
     */
    public function readAction(DiscountModule $discountModule)
    {
        return $this->createJsonSerializeResponse($discountModule, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNT_PERCENTAGE_WRITE')")
     */
    public function updateAction(Request $request, DiscountModule $discountModule)
    {
        return $this->doProcessForm($discountModule, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNT_PERCENTAGE_WRITE')")
     */
    public function deleteAction(DiscountModule $discountModule)
    {
        $this->deleteObject($discountModule);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_DISCOUNT_TARGETAMOUNT_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $discountModules = DiscountModuleQuery::create()->findPks($ids, $con);
        foreach ($discountModules as $discountModule) {
            $value = $discountModule->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $discountModule->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $discountModule->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        DiscountModuleQuery::create()->findPks($ids, $con)->delete($con);
    }
}