<?php

namespace Widget\Discount\CouponBundle\Tests\Discount;

use Backend\BaseBundle\Tests\Fixture\BaseTestCase;
use Widget\Discount\CouponBundle\Discount\CouponDiscount;
use Widget\DiscountBundle\Service\Discount\DiscountInterface;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;

class CouponDiscountTest extends BaseTestCase
{
    public function test_create_empty_config()
    {
        // arrange
        $config = array();

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_empty_discount_config()
    {
        // arrange
        $config = array(
            'config' => array(
                0 => array()
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_empty_discount_amount_config()
    {
        // arrange
        $config = array(
            'config' => array(
                0 => array(
                    'percentage' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_empty_discount_percentage_config()
    {
        // arrange
        $config = array(
            'config' => array(
                0 => array(
                    'amount' => 1,
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_empty_start_at()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'stop_at' => 'fake date',
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_empty_stop_at()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => 'fake date',
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_start_at_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => 'fake date',
            'stop_at' => new \DateTime('now'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_stop_at_not_datetime()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => new \DateTime('now'),
            'stop_at' => 'fake date',
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_stop_not_in_time_range()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => new \DateTime('2099-12-01'),
            'stop_at' => new \DateTime('2099-12-31'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'percentage' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);

        // assert
        $this->assertNull($couponDiscount);
    }

    public function test_create_ok()
    {
        // arrange
        $config = array(
            'name' => 'Discount ',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'discount' => 1
                )
            )
        );

        // act
        $couponDiscount = CouponDiscount::create($config);
        // assert
        $this->assertInstanceOf(DiscountInterface::class, $couponDiscount);
    }

    /**
     * 使用金額折扣
     */
    public function test_process_ok_1()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();

        $config = array(
            'name' => 'Discount Coupon',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'discount' => 10
                )
            )
        );
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($couponDiscount, 'config', $config);
        // act
        $couponDiscount->process($order);
        
        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(1, $order->getDiscounts());
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals($config['name'], $discountOrderItem->getProductName());
        $this->assertEquals(-10, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    /**
     * 百分比折扣
     */
    public function test_process_ok_2()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();

        $config = array(
            'name' => 'Discount Coupon',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' => array(
                0 => array(
                    'amount' => 1,
                    'discount' => '10%'
                )
            )
        );
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($couponDiscount, 'config', $config);
        // act
        $couponDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(1, $order->getDiscounts());
        /** @var OrderItem $discountOrderItem */
        $discountOrderItem = $order->getDiscounts()[0];
        $this->assertEquals($config['name'], $discountOrderItem->getProductName());
        $this->assertEquals(-1000, $discountOrderItem->getUnitPrice());
        $this->assertEquals(1, $discountOrderItem->getQuantity());
    }

    public function test_process_no_discount()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();

        $config = array(
            'name' => 'Discount Coupon',
            'start_at' => new \DateTime('2000-01-01'),
            'stop_at' => new \DateTime('2099-01-31'),
            'config' => array(
                0 => array(
                    'amount' => 10001,
                    'discount' => '10%'
                )
            )
        );
        $order = new Order();
        $orderItem = new OrderItem();
        $orderItem
            ->setUnitPrice(10000)
            ->setQuantity(1)
            ->setProductName('test_product')
            ->setProductId(1)
        ;
        $order->addProducts($orderItem);

        $this->setObjectAttribute($couponDiscount, 'config', $config);
        // act
        $couponDiscount->process($order);

        // assert
        $this->assertCount(1, $order->getProducts());
        $this->assertCount(0, $order->getDiscounts());
        $this->assertEquals(10000, $order->getOrderAmountForPay());
    }

    public function test_isPercentageOrDiscount_is_percentage()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $discountValue = "50%";
        $expectValue = array(
            'type' => 'percentage',
            'discount' => 0.5
        );

        // act
        $discountType = $this->callObjectMethod($couponDiscount, 'isPercentageOrDiscount', $discountValue);

        // assert
        $this->assertEquals($expectValue['type'], $discountType['type']);
        $this->assertEquals($expectValue['discount'], $discountType['discount']);
    }

    public function test_isPercentageOrDiscount_is_discount()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $discountValue = "50";
        $expectValue = array(
            'type' => 'amount',
            'discount' => 50
        );
        // act
        $discountType = $this->callObjectMethod($couponDiscount, 'isPercentageOrDiscount', $discountValue);

        // assert
        $this->assertEquals($expectValue['type'], $discountType['type']);
        $this->assertEquals($expectValue['discount'], $discountType['discount']);
    }

    /**
     * % + 文字
     */
    public function test_isPercentageOrDiscount_not_match_rule()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $discountValue = "AAA%%";

        // act
        $discountType = $this->callObjectMethod($couponDiscount, 'isPercentageOrDiscount', $discountValue);

        // assert
        $this->assertNull($discountType);
    }

    /**
     * 只有文字
     */
    public function test_isPercentageOrDiscount_not_match_rule_2()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $discountValue = "AAA";

        // act
        $discountType = $this->callObjectMethod($couponDiscount, 'isPercentageOrDiscount', $discountValue);

        // assert
        $this->assertNull($discountType);
    }

    /**
     * 打了兩次%%
     */
    public function test_isPercentageOrDiscount_not_match_rule_3()
    {
        // arrange
        $couponDiscount = $this->getMockBuilder(CouponDiscount::class)
            ->disableOriginalConstructor()
            ->setMethods(null)
            ->getMock();
        $discountValue = "111%%";

        // act
        $discountType = $this->callObjectMethod($couponDiscount, 'isPercentageOrDiscount', $discountValue);

        // assert
        $this->assertNull($discountType);
    }
}