<?php

namespace Widget\Discount\CouponBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Event\Controller\ConfigEvent;
use Backend\BaseBundle\Event\Controller\MenuEvent;
use Backend\BaseBundle\Event\Controller\TypeRolesEvent;
use Backend\BaseBundle\Form\Type;
use Backend\BaseBundle\Form\Type\BaseFormType;
use Backend\BaseBundle\Form\Type\DisplayType;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Routing\Router;
use Symfony\Component\Validator\Constraints;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\DiscountBundle\Model\DiscountCoupon;
use Widget\DiscountBundle\Model\DiscountCouponQuery;
use Widget\DiscountBundle\Model\DiscountModule;
use Widget\DiscountBundle\Model\DiscountModuleQuery;

/**
 * @DI\Service()
 */
class CouponListener
{
    const CONTROLLER_NAME = 'WidgetDiscountCouponBundle:Backend/Coupon';

    /** @var  Router */
    protected $router;

    /** @var  Session */
    protected $session;

    /**
     * 注入session服務
     * @InjectParams()
     */
    public function injectSession(Session $session)
    {
        $this->session = $session;
    }

    /**
     * 注入router服務
     * @InjectParams()
     */
    public function injectRouter(Router $router)
    {
        $this->router = $router;
    }

    /**
     * 設定選單
     * @DI\Observe(MenuEvent::EVENT_MENU)
     */
    public function onMenu(MenuEvent $event)
    {
        $user = $event->getUser();
        // 驗證是否有權限
        if(!$user->hasRoleOrSuperAdmin('ROLE_DISCOUNT_COUPON_READ')) {
            return;
        }

        $isActive = preg_match('/^widget_discount_coupon_backend_/i', $event->getRequestRouteName());
        $slug = $user->getSiteSlug();

        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.discount_group' => array(
                'class' => 'fa fa-money',
                'active' => $isActive,
                'target' => array(
                    'menu.discount_coupon' => array(
                        'class' => 'fa fa-money',
                        'active' => $isActive,
                        'target' => $this->router->generate('widget_discount_coupon_backend_coupon_index', array('slug' => $slug)),
                    ),
                ),
            ),
        ));
    }

    /**
     * 讓帳號管理介面加入權限設定
     * @DI\Observe(TypeRolesEvent::EVENT_TYPE_ROLES)
     */
    public function onTypeRoles(TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_DISCOUNT_COUPON', array(
            'ROLE_DISCOUNT_COUPON_READ',
            'ROLE_DISCOUNT_COUPON_WRITE',
        ));
    }

    /**
     * @DI\Observe(ConfigEvent::EVENT_CONFIG)
     */
    public function onConfig(ConfigEvent $event)
    {

        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->stopPropagation();

        $discountCoupon = array('CreatedAt', BaseType\HiddenType::class, array('label' => 'index.discount_coupon.stop_at'));
        if ( ! preg_match('/new/', $_SERVER['REQUEST_URI']) ) {
            $discountCoupon = array(
                'DiscountCoupon',  //column
                new BaseFormType(function(FormBuilderInterface $builder){
                    $builder
                        ->add('ReUse', DisplayType::class,
                            array(
                                'template' => 'WidgetDiscountCouponBundle:Backend:_random_code.html.twig',
                                'label' => ' ',
                                'required' => false,
                            )
                        )
                        ->add('CouponCode', BaseType\TextType::class,
                            array(
                                'label' => 'index.discount_coupon.coupons.code',
                                'constraints' => array(
                                    new Callback(function($value, ExecutionContextInterface $context){
                                        $hasCoupons = DiscountCouponQuery::create()
                                            ->filterByCouponCode($value)
                                            ->find();
                                        $id = explode("/", $_SERVER['REQUEST_URI'])[5];
                                        /** @var DiscountCoupon $coupon */
                                        foreach ($hasCoupons as $coupon) {
                                            if ($coupon->getId() != $id){
                                                $context->addViolation('validator.discount.coupon.code_unique');
                                            }
                                        }
                                    }),
                                ),
                            )
                        )
                    ;
                }, 'DiscountCoupon'),
                array(
                    'label' => 'index.discount_coupon.coupons.coupone',
                    'data_class' => DiscountCoupon::class,
                    'attr' => array(
                        'data-subform' => 'panel',
                    ),
                ),
            );
        }

        $event->setConfig(array(
            'model' => DiscountModule::class,
            'route' => array(
                'index' => true,
                'sort' => true,
                'new' => true,
                'edit' => true,
                'delete' => true,
            ),
            'index' => array(
                '#' => 'Id',
                'index.discount_coupon.name' => 'Name',
                'index.discount_coupon.sort' => 'Sort',
                'index.discount_coupon.start_at' => 'StartAt',
                'index.discount_coupon.stop_at' => 'StopAt',
            ),
            'form' => array(
                array(
                    'Name',
                    BaseType\TextType::class,
                    array(
                        'label' => 'index.discount_coupon.name',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Config',
                    BaseType\CollectionType::class,
                    array(
                        'label' => 'index.discount_coupon.config.config',
                        'type' => new Type\BaseFormType(function(FormBuilderInterface $builder){
                            $builder
                                ->add('amount', BaseType\IntegerType::class, array(
                                    'label' => 'index.discount_coupon.config.amount',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                        new Constraints\GreaterThan(array(
                                            'value' => 0,
                                        )),
                                    ),
                                ))
                                ->add('discount', BaseType\TextType::class, array(
                                    'label' => 'index.discount_coupon.config.discount',
                                    'constraints' => array(
                                        new Constraints\NotBlank(),
                                        new Constraints\GreaterThan(array(
                                            'value' => 0,
                                        )),
                                    ),
                                ))
                            ;
                        }, 'Config'),
                        'allow_add' => true,
                        'allow_delete'  => false,
                    ),
                ),
                array(
                    'Sort',
                    BaseType\IntegerType::class,
                    array(
                        'label' => 'index.discount_coupon.sort',
                        'constraints' => array(
                            new Constraints\NotBlank(),
                            new Constraints\GreaterThan(array(
                                'value' => 0,
                            )),
                        ),
                    )
                ),
                array(
                    'StartAt',
                    BaseType\DateType::class,
                    array(
                        'label' => 'index.discount_coupon.start_at',
                        'required' => true,
                        'widget' => 'single_text',
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'StopAt',
                    BaseType\DateType::class,
                    array(
                        'label' => 'index.discount_coupon.stop_at',
                        'required' => true,
                        'widget' => 'single_text',
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                $discountCoupon,
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'widget_discount_coupon_backend_coupon_index',
                    ),
                ),
            ),
            'security' => array(
                'new' => 'ROLE_DISCOUNT_COUPON_WRITE',
                'edit' => 'ROLE_DISCOUNT_COUPON_WRITE',
                'delete' => 'ROLE_DISCOUNT_COUPON_WRITE',
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit',
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
        ));
    }
}