<?php

namespace Widget\Discount\CouponBundle\Discount;

use Widget\DiscountBundle\Service\Discount\AbstractDiscount;
use Widget\OrderBundle\Model\Order;
use Widget\OrderBundle\Model\OrderItem;

class CouponDiscount extends AbstractDiscount
{
    public static function create($config)
    {
        if (!isset($config['config']) || !isset($config['config'][0])) {
            return null;
        }

        $discountConfig = $config['config'][0];
        if (!isset($discountConfig['amount']) || !isset($discountConfig['discount'])) {
            return null;
        }

        if (!isset($config['start_at']) || !isset($config['stop_at'])) {
            return null;
        }

        if (!$config['start_at'] instanceof \DateTime || !$config['stop_at'] instanceof \DateTime) {
            return null;
        }

        if (time() < $config['start_at']->getTimeStamp() || time() > $config['stop_at']->getTimeStamp()) {
            return null;
        }

        return parent::create($config);
    }

    public function process(Order $order, \PropelPDO $con = null)
    {
        $discountConfig = $this->config['config'][0];
        $order->syncAmount($con);
        if ($order->getAmountForPay() < $discountConfig['amount']) {
            return;
        }

        if ($this->isPercentageOrDiscount($discountConfig['discount']) == null) {
            return;
        }

        $discountMode = $this->isPercentageOrDiscount($discountConfig['discount']);

        if ($discountMode['type'] == 'amount') {
            $discountAmount = $discountMode['discount'];
        }
        elseif ($discountMode['type'] == 'percentage') {
            $discountAmount = round($order->getAmountForPay() * $discountMode['discount'], 2);
        }

        if ($order->getAmountForPay() < $discountAmount) {
            $discountPrice = $order->getAmountForPay();
        } else {
            $discountPrice = $discountAmount;
        }        

        $this->processed = true;
        $item = new OrderItem();
        $item->setOrder($order);
        $item->setQuantity(1);
        $item->setProductName("{$this->config['name']}");
        $item->setUnitPrice(-$discountPrice);
    }

    /**
     * @param $discountValue
     * @return array|null
     */
    protected function isPercentageOrDiscount($discountValue)
    {
        if (is_numeric($discountValue)) {
            return array(
                'type' => 'amount',
                'discount' => $discountValue
            );
        }

        if (strpos($discountValue, '%') !== false) {
            $discountPercentage = str_replace('%', '', $discountValue, $count);
            if (is_numeric($discountPercentage) && $count == 1) {
                return array(
                    'type' => 'percentage',
                    'discount' => round($discountPercentage / 100, 2)
                );
            } else {
                return null;
            }
        }
        
        return null;
    }
}