<?php

namespace Widget\ContactusBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Widget\ContactusBundle\Model\Contactus;
use Widget\ContactusBundle\Model\ContactusPeer;
use Widget\ContactusBundle\Model\ContactusQuery;

/**
 * @Route("/contactus")
 * @Security("has_role_or_superadmin('ROLE_CONTACTUS')")
 */
class ContactusController extends BaseBackendAPIController
{

    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            new APIFormTypeItem('username'),
            new APIFormTypeItem('email'),
            new APIFormTypeItem('phone'),
            new APIFormTypeItem('content'),
        );
    }

    /**
     * 新增
     * 聯絡我們不用從後台新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_CONTACTUS_WRITE')")
     */
    public function createAction(Request $request)
    {
        // 抽離出來做（同樣）動作 參數： new物件、content
        return $this->doProcessForm(new Contactus(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_CONTACTUS_READ')")
     */
    public function searchAction(Request $request)
    {
        return $this->doSearch($request->query->all(), ContactusQuery::create()->distinct(), ContactusPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_CONTACTUS_READ')")
     */
    public function readAction(Contactus $contactus)
    {
        return $this->createJsonSerializeResponse($contactus, array('detail'));
    }

    /**
     * 更新
     * 聯絡我們不用從後台修改
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_CONTACTUS_WRITE')")
     */
    public function updateAction(Request $request, Contactus $contactus)
    {
        // 抽離出來做（同樣）動作 參數： 物件、content
        return $this->doProcessForm($contactus, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_CONTACTUS_WRITE')")
     */
    public function deleteAction(Contactus $contactus)
    {
        $this->deleteObject($contactus);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_CONTACTUS_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     * @return void
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $contactuss = ContactusQuery::create()->findPks($ids, $con);
        foreach ($contactuss as $contactus) {
            $value = $contactus->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $contactus->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $contactus->save($con);
        }
    }

    /**
     * 批次刪除處理
     * @param $ids
     * @param $con
     * @return void
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        ContactusQuery::create()->findPks($ids, $con)->delete($con);
    }

    /**
     * 批次設定值
     * @param $ids
     * @param $column
     * @param $value
     * @param \PropelPDO $con
     * @return void
     */
    protected function batchValue($ids, $column, $value, \PropelPDO $con)
    {
        $contactuss = ContactusQuery::create()->findPks($ids, $con);
        foreach ($contactuss as $contactus) {
            $contactus->setByName($column, $value, \BasePeer::TYPE_FIELDNAME);
            $contactus->save($con);
        }
    }
}