<?php
namespace Widget\CategoryBundle\Controller\Backend;

use Backend\BaseBundle\Model\Site;
use Symfony\Bundle\FrameworkBundle\Controller\Controller;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\ParamConverter;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Template;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Widget\CategoryBundle\Event\CategoryEvent;
use Widget\CategoryBundle\Model;
use Symfony\Component\HttpFoundation\JsonResponse;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Security\Core\Exception\AccessDeniedException;

/**
 *
 * @Route("/category")
 * @Security("has_role_or_superadmin('ROLE_CATEGORY_READ')")
 */
class CategoryController extends Controller
{
    /**
     * @Route("/{thread}", requirements={"thread" = "[a-z_]+"})
     * @Method({"GET"})
     * @Template()
     */
    public function indexAction($thread)
    {
        $event = new CategoryEvent($thread);
        $this->get('event_dispatcher')->dispatch(CategoryEvent::EVENT_CATEGORY, $event);

        if(!$event->hasThread()){
            throw $this->createNotFoundException();
        }
        return array(
            'thread' => $thread,
            'category' => $this->makeRootTypeCategory($thread),
            'maxDepth' => $event->getMaxDepth(),
            'extendTemplate' => $event->getExtendTemplate(),
        );
    }

    /**
     * @return Model\Category
     */
    protected function makeRootTypeCategory($thread)
    {
        $site = $this->get('session')->get('Site');
        $rootCategory = Model\CategoryQuery::create()
            ->filterByTreeLevel(0)
            ->useCategoryThreadQuery()
            ->filterByThread($thread)
            ->filterBySiteId($site->getId())
            ->endUse()
            ->findOne()
            ;
        if(!$rootCategory){
            $categoryThread = new Model\CategoryThread();
            $categoryThread->setSiteId($site->getId());
            $categoryThread->setThread($thread);
            $rootCategory = new Model\Category();
            $rootCategory->setSiteId($site->getId());
            $rootCategory->setCategoryThread($categoryThread);
            $rootCategory->makeRoot();
            $rootCategory->save();
            return $rootCategory;
        }
        return $rootCategory;
    }

    /**
     * @Route("/{id}", requirements={"id" = "\d+"})
     * @Method({"GET"})
     */
    public function readCategoryAction(Request $request, Model\Category $category)
    {
        return new JsonResponse($category->getAllChildrenArray());
    }

    /**
     *
     * @Route("/{id}", requirements={"id" = "\d+"})
     * @ParamConverter("site", options={"exclude": {"id"}})
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_CATEGORY_WRITE')")
     */
    public function createCategoryAction(Request $request, Model\Category $category, Site $site)
    {
        if($category->getSiteId() != $site->getId()){
            throw new AccessDeniedException();
        }

        $param = json_decode($request->getContent(), true);
        if($param['name'] == null){
            return;
        }
        $newCategory = new Model\Category();
        $newCategory->setSiteId($category->getSiteId());
        $newCategory->setName($param['name']);
        $newCategory->setStatus((bool)$param['status']);
        $newCategory->insertAsLastChildOf($category);
        $this->get("widget.category.operationlog")
            ->logOperation($newCategory);
        $newCategory->save();
        return new JsonResponse(array(
            'id' => $newCategory->getId(),
            'status' => $newCategory->getStatus(),
            'name' => $newCategory->getName(),
            'level' => $newCategory->getLevel(),
        ));
    }

    /**
     *
     * @Route("/{id}", requirements={"id" = "\d+"})
     * @ParamConverter("site", options={"exclude": {"id"}})
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_CATEGORY_WRITE')")
     */
    public function deleteCategoryAction(Request $request, Model\Category $category, Site $site)
    {
        if($category->getSiteId() != $site->getId()){
            throw new AccessDeniedException();
        }
        $category->delete();
        $this->get("widget.category.operationlog")
            ->logOperation($category);
        return new JsonResponse(array(
            'id' => $category->getId(),
        ));
    }

    /**
     * @Route("/{id}", requirements={"id" = "\d+"})
     * @ParamConverter("site", options={"exclude": {"id"}})
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_CATEGORY_WRITE')")
     */
    public function updateCategoryAction(Request $request, Model\Category $category, Site $site)
    {
        if($category->getSiteId() != $site->getId()){
            throw new AccessDeniedException();
        }
        $param = json_decode($request->getContent(), true);
        if($param['name'] == null){
            return;
        }
        $category->setName($param['name']);
        $category->setStatus((bool)$param['status']);
        $this->get("widget.category.operationlog")
            ->logOperation($category);
        $category->save();
        return new JsonResponse(array(
            'id' => $category->getId(),
            'status' => $category->getStatus(),
            'name' => $category->getName(),
            'level' => $category->getLevel(),
        ));
    }

    /**
     * @Route("/{id}/moveto", requirements={"id" = "\d+"})
     * @ParamConverter("site", options={"exclude": {"id"}})
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_CATEGORY_WRITE')")
     */
    public function movetoCategoryAction(Request $request, Model\Category $category, Site $site)
    {
        $this->filterCategory($site, $category);
        $param = json_decode($request->getContent(), true);
        $targetCategory = $this->filterCategory($site, Model\CategoryQuery::create()->findPk($param['target_id']));
        $isInsert = $param['position'] == 'inside';

        if($targetCategory->getThreadId() != $category->getThreadId()){
            throw new AccessDeniedException();
        }

        if($isInsert){
            $category->moveToFirstChildOf($targetCategory);
        }
        else{
            $category->moveToNextSiblingOf($targetCategory);
        }
        $this->get("widget.category.operationlog")
            ->logOperation($category);
        $category->save();
        return new JsonResponse(array(
            'id' => $category->getId(),
        ));
    }

    /**
     * @param Site $site
     * @param Model\Category|null $category
     * @return Model\Category
     */
    protected function filterCategory(Site $site, Model\Category $category = null)
    {
        if(!$category || $category->getSiteId() != $site->getId()){
            throw new AccessDeniedException();
        }
        return $category;
    }
}