<?php

namespace Widget\BannerBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\Form\Extension\Core\Type\IntegerType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\Validator\Constraints\GreaterThan;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Range;
use Widget\BannerBundle\Model\Banner;
use Widget\BannerBundle\Model\BannerPeer;
use Widget\BannerBundle\Model\BannerQuery;
use Widget\PhotoBundle\Form\Type\APIPhotoListType;

/**
 * @Route("/banner")
 * @Security("has_role_or_superadmin('ROLE_BANNER')")
 */
class BannerController extends BaseBackendAPIController
{
    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('banner_id'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                )
            )),
            new APIFormTypeItem('url'),
            new APIFormTypeItem('target'),
            (new APIFormTypeItem('sort'))->setOptions(array(
                'constraints' => array(
                    new GreaterThan(array(
                        'value' => -1,
                        'message' => "error.banner.greater_than_0",
                    ))
                ),
            ))->setFieldType(IntegerType::class),
            new APIFormTypeItem('status'),
            new APIFormTypeItem('start_at'),
            new APIFormTypeItem('stop_at'),
            (new APIFormTypeItem('photos'))->setFieldType(new APIPhotoListType()),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function createAction(Request $request)
    {
        // 抽離出來做（同樣）動作 參數： 物件、content
        return $this->doProcessForm(new Banner(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = BannerQuery::create()
            ->joinWithBannerCategory()
            ->useBannerCategoryQuery()
                ->joinWithPhotoConfig()
            ->endUse()
            ->distinct();
        return $this->doSearch($request->query->all(), $query, BannerPeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_READ')")
     */
    public function readAction(Banner $banner)
    {
        return $this->createJsonSerializeResponse($banner, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function updateAction(Request $request, Banner $banner)
    {
        return $this->doProcessForm($banner, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function deleteAction(Banner $banner)
    {
        $this->deleteObject($banner);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $banners = BannerQuery::create()->findPks($ids, $con);
        foreach ($banners as $banner) {
            $value = $banner->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $banner->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $banner->save($con);
        }
    }

    /**
     * 批次設定值
     * @param $ids
     * @param $column
     * @param $value
     * @param \PropelPDO $con
     * @return \Symfony\Component\HttpFoundation\JsonResponse|void
     */
    protected function batchValue($ids, $column, $value, \PropelPDO $con)
    {
        $banners = BannerQuery::create()->findPks($ids, $con);
        foreach ($banners as $banner) {
            $banner->setByName($column, $value, \BasePeer::TYPE_FIELDNAME);
            $banner->save($con);
        }
    }    
    
    /**
     * 批次刪除
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        BannerQuery::create()->findPks($ids, $con)->delete($con);
    }
}