<?php

namespace Widget\BannerBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\BannerBundle\Model\BannerCategory;
use Widget\BannerBundle\Model\BannerCategoryPeer;
use Widget\BannerBundle\Model\BannerCategoryQuery;

/**
 * @Route("/bannercategory")
 * @Security("has_role_or_superadmin('ROLE_BANNER')")
 */
class BannerCategoryController extends BaseBackendAPIController
{
    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('name'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                    new Regex(array(
                        'pattern' => '/^[0-9a-z_]+$/',
                        'message' => 'error.banner_category.name.format'
                    )),
                    new Callback(function($value, ExecutionContextInterface $context){
                        $object = $context->getRoot()->getData();
                        $bannerCategory = BannerCategoryQuery::create()
                            ->filterByName($value)
                            ->findOne();
                        if($bannerCategory && $bannerCategory->getId() != $object->getId()){
                            $context->addViolation('error.banner_category.type.duplicate');
                        }
                    }),
                ),
            )),
            (new APIFormTypeItem('title'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                ),
            )),
            new APIFormTypeItem('photo_config_id'),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function createAction(Request $request)
    {
        // 抽離出來做（同樣）動作 參數： 物件、content
        return $this->doProcessForm(new BannerCategory(), $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = BannerCategoryQuery::create()->joinWithPhotoConfig()->distinct();
        return $this->doSearch($request->query->all(), $query, BannerCategoryPeer::class);
    }

    /**
     * 表單列表
     * @Route("s/all")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_READ')")
     */
    public function listAction(Request $request)
    {
        $bannerCategory = BannerCategoryQuery::create()->find();
        if (!$bannerCategory){
            return new Response('', Response::HTTP_NOT_FOUND);
        }
        return $this->createJsonSerializeResponse($bannerCategory, array('list'));
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_READ')")
     */
    public function readAction(BannerCategory $bannerCategory)
    {
        return $this->createJsonSerializeResponse($bannerCategory, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function updateAction(Request $request, BannerCategory $bannerCategory)
    {
        return $this->doProcessForm($bannerCategory, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function deleteAction(BannerCategory $bannerCategory)
    {
        $this->deleteObject($bannerCategory);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_BANNER_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $bannerCategories = BannerCategoryQuery::create()->findPks($ids, $con);
        foreach ($bannerCategories as $bannerCategory) {
            $value = $bannerCategory->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $bannerCategory->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $bannerCategory->save($con);
        }
    }

    /**
     * 批次刪除
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        BannerCategoryQuery::create()->findPks($ids, $con)->delete($con);
    }
}