<?php
namespace Widget\BannerBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Event\Controller\ConfigEvent;
use Backend\BaseBundle\Event\Controller\MenuEvent;
use Backend\BaseBundle\Event\Controller\TypeRolesEvent;
use Backend\BaseBundle\Form\Type\RelationType;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation\InjectParams;
use JMS\DiExtraBundle\Annotation\Observe;
use JMS\DiExtraBundle\Annotation\Service;
use Symfony\Bundle\FrameworkBundle\Routing\Router;
use Symfony\Component\Form\Extension\Core\Type\TextType;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\BannerBundle\Model\Banner;
use Widget\BannerBundle\Model\BannerQuery;
use Widget\PhotoBundle\Model\PhotoConfig;

/**
 * @Service
 */
class BannerListener
{
    const CONTROLLER_NAME = 'WidgetBannerBundle:Backend/Banner';

    /** @var Router */
    protected $router;

    /** @var Session */
    protected $session;

    /**
     * 注入session服務
     * @InjectParams()
     */
    public function injectSession(Session $session)
    {
        $this->session = $session;
    }

    /**
     * @InjectParams()
     */
    public function injectRouter(Router $router)
    {
        $this->router = $router;
    }

    /**
     * @Observe(MenuEvent::EVENT_MENU)
     */
    public function onMenu(MenuEvent $event)
    {
        $user = $event->getUser();
        if(!$user->hasRoleOrSuperAdmin('ROLE_BANNER_CATEGORY_READ')){
            return;
        }
        
        $isActive = preg_match('/^widget_banner_backend_banner_/i', $event->getRequestRouteName());
        $slug = $user->getSiteSlug();
        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.banner' => array(
                'class' => 'fa fa-clone',
                'active' => $isActive,
                'target' => array(
                    'menu.category_list' => array(
                        'class' => 'fa fa-clone',
                        'active' => $isActive,
                        'target' => $this->router->generate('widget_banner_backend_banner_index', array('slug' => $slug)),
                    ),
                ),
            ),
        ));
    }

    /**
     * @Observe(TypeRolesEvent::EVENT_TYPE_ROLES)
     */
    public function onTypeRoles(TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_BANNER', array(
            'ROLE_BANNER_CATEGORY_READ',
            'ROLE_BANNER_CATEGORY_WRITE',
        ));
    }

    /**
     * @Observe(ConfigEvent::EVENT_CONFIG)
     */
    public function onConfig(ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->setConfig(array(
            'model' => Banner::class,
            'route' => array(
                'sort' => true,
                'index' => true,
                'edit' => true,
                'new' => true,
                'delete' => true,
                'search' => true,
            ),
            'index' => array(
                '#' => 'Id',
                'index.category.name' => 'Name',
                'index.category.title' => 'Title',
                'index.category.photo_config' => 'PhotoConfig',
                'index.category.created_at' => 'CreatedAt',
                'index.category.updated_at' => 'UpdatedAt',
            ),
            'security' => array(
                'new' => 'ROLE_BANNER_CATEGORY_WRITE',
                'edit' => 'ROLE_BANNER_CATEGORY_WRITE',
                'delete' =>'ROLE_BANNER_CATEGORY_WRITE',
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
            'search' => array(
                array(
                    'Name',
                    TextType::class,
                    array(
                        'label' => 'index.category.name',
                        'required' => false,
                    ),
                ),
                array(
                    'Title',
                    TextType::class,
                    array(
                        'label' => 'index.category.title',
                        'required' => false,
                    ),
                ),
            ),
            'form' => array(
                array(
                    'Name',
                    TextType::class,
                    array(
                        'label' => 'form.label.edit.category.name',
                        'help' => 'form.help.category.name',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                            new Regex(array(
                                'pattern' => '/^[0-9a-z_]+$/',
                            )),
                            new Callback(function($value, ExecutionContextInterface $context){
                                $object = $context->getRoot()->getData();
                                $banner = BannerQuery::create()
                                    ->filterBySiteId($this->session->get('Site')->getId())
                                    ->filterByName($value)
                                    ->findOne();
                                if($banner && $banner->getId() != $object->getId()){
                                    $context->addViolation('error.banner.type.duplicate');
                                }
                            }),
                        ),
                    ),
                ),
                array(
                    'Title',
                    TextType::class,
                    array(
                        'label' => 'form.label.edit.category.title',
                        'required' => true,
                    ),
                ),
                //關聯到photo_config的資料表
                array(
                    'PhotoConfig',
                    RelationType::class,
                    array(
                        'label' => 'form.label.edit.category.photo_config',
                        'class' => PhotoConfig::class,
                        'multiple' => false,
                        'required' => true,
                    ),
                ),
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'widget_banner_backend_banner_index',
                    ),
                ),
            ),
        ));
    }
}