<?php

namespace Widget\ArticleBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Backend\BaseBundle\Propel\I18n;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\ArticleBundle\Model\Article;
use Widget\ArticleBundle\Model\ArticlePeer;
use Widget\ArticleBundle\Model\ArticleQuery;

/**
 * @Route("/article")
 * @Security("has_role_or_superadmin('ROLE_ARTICLE')")
 */
class ArticleController extends BaseBackendAPIController
{

    /**
     * 欄位表單設定
     * 主要是USER送過來的欄位
     * @return APIFormTypeItem[]
     */
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('type'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                    new Regex(array(
                        'pattern' => '/^[0-9a-z]+$/',
                        'message' => 'error.article.type.format_incorrect_field'
                    )),
                    new Callback(function($value, ExecutionContextInterface $context){
                        $object = $context->getRoot()->getData();
                        $article = ArticleQuery::create()
                            ->filterByType($value)
                            ->findOne();
                        if($article && $article->getId() != $object->getId()){
                            $context->addViolation('error.article.type.duplicate');
                        }
                    })
                )
            )),
            (new APIFormTypeItem('title'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.article.title.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('brief'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.article.brief.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('content'))->setOptions(array(
                'constraints' => array(
                    new NotBlank(array(
                        'message' => 'error.article.content.missing_field',
                    )),
                ),
            )),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function createAction(Request $request)
    {
        $article = new Article();
        if ($article instanceof I18n){
            $locale = $request->query->get('_locale');
            $article->setLocale($locale);
        }
        // 抽離出來做（同樣）動作 參數： 物件、content
        return $this->doProcessForm($article, $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = ArticleQuery::create();
        if ($query instanceof I18n){
            $locale = $request->query->get('_locale');
            $query->joinArticleI18n($locale);
        }
        return $this->doSearch($request->query->all(), $query->distinct(), ArticlePeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_READ')")
     */
    public function readAction(Request $request, Article $article)
    {
        if ($article instanceof I18n){
            $locale = $request->query->get('_locale');
            $article->setLocale($locale);
        }
        return $this->createJsonSerializeResponse($article, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function updateAction(Request $request, Article $article)
    {
        if ($article instanceof I18n){
            $locale = $request->query->get('_locale');
            $article->setLocale($locale);
        }
        return $this->doProcessForm($article, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function deleteAction(Article $article)
    {
        $this->deleteObject($article);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $articles = ArticleQuery::create()->findPks($ids, $con);
        foreach ($articles as $article) {
            $value = $article->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $article->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $article->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        ArticleQuery::create()->findPks($ids, $con)->delete($con);
    }
}