<?php
namespace Widget\ArticleBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Event\Controller\ConfigEvent;
use Backend\BaseBundle\Event\Controller\MenuEvent;
use Backend\BaseBundle\Event\Controller\TypeRolesEvent;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation as DI;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Symfony\Component\HttpFoundation\Session\Session;
use Symfony\Component\Routing\Router;
use Symfony\Component\Validator\Constraints\Callback;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\ArticleBundle\Model\ArticleQuery;
use Widget\PostBundle\Form\Type\RichTextType;

/**
 * 監聽或做任何事件時，需先將本身listener註冊為服務
 * @DI\Service()
 */
class ArticleListener
{
    const CONTROLLER_NAME = 'WidgetArticleBundle:Backend/Article';

    /** @var  Router */
    protected $router;

    /** @var  Session */
    protected $session;

    /**
     * 注入session服務
     * @InjectParams()
     */
    public function injectSession(Session $session)
    {
        $this->session = $session;
    }

    /**
     * 注入router服務
     * @InjectParams()
     */
    public function injectRouter(Router $router)
    {
        $this->router = $router;
    }

    /**
     * 設定選單
     * @DI\Observe(MenuEvent::EVENT_MENU)
     */
    public function onMenu(MenuEvent $event)
    {
        $user = $event->getUser();
        // 驗證是否有權限
        if(!$user->hasRoleOrSuperAdmin('ROLE_ARTICLE_READ')) {
            return;
        }

        $isActive = preg_match('/^widget_article_backend_/i', $event->getRequestRouteName());
        $slug = $user->getSiteSlug();
        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.article' => array(
                'class' => 'fa fa-book',
                'active' => $isActive,
                'target' => $this->router->generate('widget_article_backend_article_index', array('slug' => $slug)),
            ),
        ));
    }

    /**
     * 讓帳號管理介面加入權限設定
     * @DI\Observe(TypeRolesEvent::EVENT_TYPE_ROLES)
     */
    public function onTypeRoles(TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_ARTICLE', array(
            'ROLE_ARTICLE_READ',
            'ROLE_ARTICLE_WRITE',
        ));
    }

    /**
     * @DI\Observe(ConfigEvent::EVENT_CONFIG)
     */
    public function onConfig(ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->setConfig(array(
            'route' => array(
                'index' => true,
                'sort' => true,
                'new' => true,
                'edit' => true,
                'delete' => true,
            ),
            'index' => array(
                '#' => 'Id',
                'index.article.type' => 'Type',
                'index.article.title' => 'Title',
                'index.article.created_at' => 'CreatedAt',
                'index.article.updated_at' => 'UpdatedAt',
            ),
            'form' => array(
                array(
                    'Type',
                    BaseType\TextType::class,
                    array(
                        'label' => 'index.article.type',
                        'help' => 'form.help.type',
                        'constraints' => array(
                            new NotBlank(),
                            new Regex(array(
                                'pattern' => '/^[0-9a-z_]+$/',
                            )),
                            new Callback(function($value, ExecutionContextInterface $context){
                                $object = $context->getRoot()->getData();
                                $article = ArticleQuery::create()
                                    ->filterBySiteId($this->session->get('Site')->getId())
                                    ->filterByType($value)
                                    ->findOne();
                                if($article && $article->getId() != $object->getId()){
                                    $context->addViolation('error.article.type.duplicate');
                                }
                            }),
                        ),
                    ),
                ),
                array(
                    'Title',
                    BaseType\TextType::class,
                    array(
                        'label' => 'index.article.title',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Brief',
                    BaseType\TextareaType::class,
                    array(
                        'label' => 'index.article.brief',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(array(
                                'message' => 'error.missing_field',
                            )),
                        ),
                    ),
                ),
                array(
                    'Content',
                    RichTextType::class,
                    array(
                        'label' => 'index.article.content',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'widget_article_backend_article_index',
                    ),
                ),
            ),
            'security' => array(
                'new' => 'ROLE_ARTICLE_WRITE',
                'edit' => 'ROLE_ARTICLE_WRITE',
                'delete' => 'ROLE_ARTICLE_WRITE',
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
        ));
    }
}