<?php

namespace Widget\ArticleBundle\Controller\BackendAPI;

use Backend\BaseBundle\Controller\BackendAPI\BaseBackendAPIController;
use Backend\BaseBundle\Form\Type\APIFormTypeItem;
use Backend\BaseBundle\Model\SiteConfigQuery;
use Backend\BaseBundle\Propel\I18n;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Method;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Route;
use Sensio\Bundle\FrameworkExtraBundle\Configuration\Security;
use Symfony\Component\Form\Extension\Core\Type\ChoiceType;
use Symfony\Component\HttpFoundation\Request;
use Symfony\Component\HttpFoundation\Response;
use Symfony\Component\Validator\Constraints as Assert;
use Symfony\Component\Validator\Context\ExecutionContextInterface;
use Widget\ArticleBundle\Model\Article;
use Widget\ArticleBundle\Model\ArticlePeer;
use Widget\ArticleBundle\Model\ArticleQuery;
use Widget\PhotoBundle\Form\Type\APIPhotoListType;

/**
 * @Route("/article")
 * @Security("has_role_or_superadmin('ROLE_ARTICLE')")
 */
class ArticleController extends BaseBackendAPIController
{
    protected function getFormConfig()
    {
        return array(
            (new APIFormTypeItem('type'))->setOptions(array(
                'constraints' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.missing_field',
                    )),
                    new Assert\Regex(array(
                        'pattern' => '/^[0-9a-z]+$/',
                        'message' => 'error.article.type.format_incorrect_field'
                    )),
                    new Assert\Callback(function($value, ExecutionContextInterface $context){
                        $obj = $context->getRoot()->getData();
                        $article = ArticleQuery::create()->findOneByType($value);
                        if($article && $obj->getId() != $article->getId()){
                            $context->addViolation('error.article.type.duplicate');
                        }
                    })
                )
            )),
            (new APIFormTypeItem('title'))->setOptions(array(
                'constraints' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.article.title.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('brief'))->setOptions(array(
                'constraints' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.article.brief.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('content'))->setOptions(array(
                'constraints' => array(
                    new Assert\NotBlank(array(
                        'message' => 'error.article.content.missing_field',
                    )),
                )
            )),
            (new APIFormTypeItem('photos'))->setFieldType(new APIPhotoListType()),
            (new APIFormTypeItem('data_status'))->setFieldType(ChoiceType::class)->setOptions(array(
                'choices' => array(
                    true => 1,
                    false => 0,
                    null => 0,
                )
            )),
        );
    }

    /**
     * 新增
     * @Route("s")
     * @Method({"POST"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function createAction(Request $request)
    {
        if (!($locale = $request->attributes->get('_locale'))){
            return $this->createHttpExceptionResponse(Response::HTTP_BAD_REQUEST);
        }
        $parameter = json_decode($request->getContent(), true);
        if (!($article = ArticleQuery::create()->findOneByType($parameter['type']))) {
            $article = new Article();
        }
        $article->setLocale($locale);
        return $this->doProcessForm($article, $request->getContent());
    }

    /**
     * 搜尋
     * @Route("s")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_READ')")
     */
    public function searchAction(Request $request)
    {
        $query = ArticleQuery::create()->joinWithI18n($request->attributes->get('_defaultLocale'));
        return $this->doSearch($request->query->all(), $query->distinct(), ArticlePeer::class);
    }

    /**
     * 讀取
     * @Route("/{id}")
     * @Method({"GET"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_READ')")
     */
    public function readAction(Request $request, Article $article)
    {
        if (!($locale = $request->attributes->get('_locale'))){
            return $this->createHttpExceptionResponse(Response::HTTP_BAD_REQUEST);
        }
        $article->setLocale($locale);
        return $this->createJsonSerializeResponse($article, array('detail'));
    }

    /**
     * 更新
     * @Route("/{id}")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function updateAction(Request $request, Article $article)
    {
        if (!($locale = $request->attributes->get('_locale'))){
            return $this->createHttpExceptionResponse(Response::HTTP_BAD_REQUEST);
        }
        $article->setLocale($locale);
        return $this->doProcessForm($article, $request->getContent());
    }

    /**
     * 刪除
     * @Route("/{id}")
     * @Method({"DELETE"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function deleteAction(Article $article)
    {
        $this->deleteObject($article);
        return $this->createJsonResponse();
    }

    /**
     * 批次管理
     * @Route("s")
     * @Method({"PUT"})
     * @Security("has_role_or_superadmin('ROLE_ARTICLE_WRITE')")
     */
    public function batchAction(Request $request)
    {
        return parent::batchAction($request);
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchSwitch($ids, $column, \PropelPDO $con)
    {
        $articles = ArticleQuery::create()->findPks($ids, $con);
        foreach ($articles as $article) {
            $value = $article->getByName($column, \BasePeer::TYPE_FIELDNAME);
            $article->setByName($column, !$value, \BasePeer::TYPE_FIELDNAME);
            $article->save($con);
        }
    }

    /**
     * 批次切換狀態處理
     * @param $ids
     * @param $con
     * @param $column
     */
    protected function batchDelete($ids, \PropelPDO $con)
    {
        ArticleQuery::create()->findPks($ids, $con)->delete($con);
    }
}