<?php
namespace Backend\BaseBundle\Tests\Controller\API;


use Backend\BaseBundle\Model\SiteConfig;
use Backend\BaseBundle\Model\SiteConfigQuery;
use Backend\BaseBundle\Model\SiteQuery;
use Backend\BaseBundle\Tests\Fixture\BackendWebTestCase;
use Symfony\Component\HttpFoundation\Response;

/**
 * @group functional
 */
class SEOControllerTest extends BackendWebTestCase
{
    private $originConfigs;

    public function test_indexAction_bad_request()
    {
        //arrange
        $slug = 'dgfactor';

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('backend_base_api_seo_index', array('slug' => $slug))
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_BAD_REQUEST, $response->getStatusCode());
    }

    public function test_indexAction_parse_url_fail_bad_url()
    {
        //arrange
        $site = SiteQuery::create()->findOneBySlug('dgfactor');
        $url = 'http:aksksajhkjsa';

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('backend_base_api_seo_index', array('slug' => $site->getSlug(), 'url' => $url))
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_NOT_ACCEPTABLE, $response->getStatusCode());
    }

    public function test_indexAction_parse_url_fail_bad_sheme()
    {
        //arrange
        $site = SiteQuery::create()->findOneBySlug('dgfactor');
        $url = 'abc://aksksajhkjsa/';

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('backend_base_api_seo_index', array('slug' => $site->getSlug(), 'url' => $url))
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_NOT_ACCEPTABLE, $response->getStatusCode());
    }

    public function test_indexAction_bad_domains()
    {
        //arrange
        $site = SiteQuery::create()->findOneBySlug('dgfactor');
        $url = 'http://badsite.com/';

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('backend_base_api_seo_index', array('slug' => $site->getSlug(), 'url' => $url))
        );
        $response = $this->client->getResponse();

        //assert
        $this->assertEquals(Response::HTTP_FORBIDDEN, $response->getStatusCode());
    }

    public function test_indexAction_ok()
    {
        //arrange
        $site = SiteQuery::create()->findOneBySlug('dgfactor');
        $siteConfig = SiteConfigQuery::create()->findPk(array(
            $site->getId(),
            'system'
        ));
        $configs = $this->originConfigs;
        $configs['domains'] = 'localhost,www.google.com';
        $siteConfig->setConfig($configs);
        $siteConfig->save();
        $url = 'https://www.google.com/';

        //act
        $this->client->request(
            'GET',
            $this->generateUrl('backend_base_api_seo_index', array('slug' => $site->getSlug(), 'url' => $url))
        );
        $response = $this->client->getResponse();
        $result = json_decode($response->getContent(), true);

        //assert
        $this->assertTrue($response->isOk(), $response->getStatusCode());
        $this->assertTrue($result['status']);
        $this->assertTrue(strpos($result['content'], 'Google') !== false);
    }

    public function setUp()
    {
        parent::setUp(); // TODO: Change the autogenerated stub
        $site = SiteQuery::create()->findOneBySlug('dgfactor');
        $siteConfig = SiteConfigQuery::create()->findPk(array(
            $site->getId(),
            'system'
        ));
        $this->originConfigs = $siteConfig->getConfig();
    }

    protected function tearDown()
    {
        $site = SiteQuery::create()->findOneBySlug('dgfactor');
        $siteConfig = SiteConfigQuery::create()->findPk(array(
            $site->getId(),
            'system'
        ));
        $siteConfig->setConfig($this->originConfigs);
        $siteConfig->save();
        parent::tearDown(); // TODO: Change the autogenerated stub
    }
}