<?php
namespace Backend\BaseBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Form\Type\BaseFormType;
use Backend\BaseBundle\Form\Type\DateTimeRangeType;
use Backend\BaseBundle\Form\Type\RelationType;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation\Service;
use JMS\DiExtraBundle\Annotation\Tag;
use JMS\DiExtraBundle\Annotation\Inject;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Backend\BaseBundle\Event\Controller;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Backend\BaseBundle\Model;
use Symfony\Component\Form\FormBuilderInterface;
use Symfony\Component\Form\FormInterface;
use Symfony\Component\Routing\Router;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Symfony\Component\Validator\Constraints\NotBlank;
use Symfony\Component\Validator\Constraints\Regex;
use Widget\PhotoBundle\Form\Constraints\Image;
use Widget\PhotoBundle\Form\Type\PhotoUploadType;
use Widget\PhotoBundle\Model\Photo;

/**
 * @Service
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\ConfigEvent::EVENT_CONFIG, "method": "onConfig"})
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\TypeRolesEvent::EVENT_TYPE_ROLES, "method": "onTypeRoles"})
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\MenuEvent::EVENT_MENU, "method": "onMenu"})
 */
class SiteUserListener
{
    const CONTROLLER_NAME = 'BackendBaseBundle:Backend/SiteUser';

    /** @var  EventDispatcherInterface */
    protected $eventDispatcher;

    /** @var  Router */
    protected $router;

    /**
     * @InjectParams()
     */
    public function __construct(EventDispatcherInterface $eventDispatcher, Router $router)
    {
        $this->eventDispatcher = $eventDispatcher;
        $this->router = $router;
    }

    public function onMenu(Controller\MenuEvent $event)
    {
        $isActive = preg_match('/^backend_base_backend_siteuser_/i', $event->getRequestRouteName());
        $user = $event->getUser();
        if(!$user->hasRole('ROLE_SUPERADMIN')){
            return;
        }
        $slug = $user->getSiteSlug();
        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.site_user' => array(
                'class' => 'fa fa-user',
                'active' => $isActive,
                'target' => array(
                    'menu.site_user' => array(
                        'class' => 'fa fa-user',
                        'active' => $isActive,
                        'target' => $this->router->generate('backend_base_backend_siteuser_index', array('slug' => $slug)),
                    ),
                ),
            ),
        ));
    }

    public function onTypeRoles(Controller\TypeRolesEvent $event)
    {
        $event->addTypeRoles('ROLE_SUPERADMIN', array(
            'ROLE_SUPERADMIN',
        ));
    }

    public function onConfig(Controller\ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->stopPropagation();

        $roleTypeEvent = new Controller\TypeRolesEvent();
        $this->eventDispatcher->dispatch($roleTypeEvent::EVENT_TYPE_ROLES, $roleTypeEvent);
        $event->setConfig(array(
            'route' => array(
                'sort' => true,
                'index' => true,
                'edit' => true,
                'new' => true,
                'search' => true,
                'delete' => true,
            ),
            'index' => array(
                '＃' => 'Id',
                'index.site_user.login_name' => 'LoginName',
                'index.site_user.email' => 'Email',
                'index.site_user.enable' => 'Enabled',
                'index.site_user.last_login' => 'LastLogin',
                'index.site_user.created_at' => 'CreatedAt',
            ),
            'search' => array(
                array(
                    'LoginName',
                    BaseType\TextType::class,
                    array(
                        'label' => 'search.site_user.login_name',
                        'required' => false,
                    ),
                ),
                array(
                    'Email',
                    BaseType\TextType::class,
                    array(
                        'label' => 'search.site_user.email',
                        'required' => false,
                    ),
                ),
                array(
                    'Enabled',
                    BaseType\ChoiceType::class,
                    array(
                        'required' => false,
                        'label' => 'search.site_user.enable',
                        'multiple' => false,
                        'expanded' => true,
                        'choices' => array(
                            true => 'form.choice.enable',
                            false => 'form.choice.disable',
                        ),
                    ),
                ),
                array(
                    'CreatedAt',
                    DateTimeRangeType::class,
                    array(
                        'label' => 'search.site_user.created_at',
                        'required' => false,
                        'with_seconds' => true,
                    ),
                ),
            ),
            'security' => array(
                'new' =>  'ROLE_SUPERADMIN',
                'edit' => 'ROLE_SUPERADMIN',
                'delete' => 'ROLE_SUPERADMIN',
            ),
            'batch' => array(
                'batch.site_user.enable' => array(
                    'url' => 'backend_base_backend_siteuser_batchstatustrue',
                    'security' => 'ROLE_SUPERADMIN',
                ),
                'batch.site_user.disable' => array(
                    'url' => 'backend_base_backend_siteuser_batchstatusfalse',
                    'security' => 'ROLE_SUPERADMIN',
                ),
                'batch.site_user.delete' => array(
                    'url' => 'backend_base_backend_siteuser_batchdelete',
                    'security' => 'ROLE_SUPERADMIN',
                ),
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
            'form' => array(
                array(
                    'LoginName',
                    BaseType\TextType::class,
                    array(
                        'label' => 'form.label.edit.site_user.login_name',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                            new Regex(array(
                                'pattern' => '/^[a-z][0-9a-z]+$/',
                                'message' => 'fos_user.loginName.pattern.error',
                            )),
                        ),
                    ),
                ),
                array(
                    'FirstName',
                    BaseType\TextType::class,
                    array(
                        'label' => 'form.label.edit.site_user.first_name',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'LastName',
                    BaseType\TextType::class,
                    array(
                        'label' => 'form.label.edit.site_user.last_name',
                        'required' => true,
                        'constraints' => array(
                          new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Email',
                    BaseType\EmailType::class,
                    array(
                        'label' => 'form.label.edit.site_user.email',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'Enabled',
                    BaseType\ChoiceType::class,
                    array(
                        'label' => 'form.label.edit.site_user.enable',
                        'multiple' => false,
                        'expanded' => true,
                        'choices' => array(
                            true => 'form.choice.enable',
                            false => 'form.choice.disable',
                        ),
                    ),
                ),
                array(
                    'SiteGroups',
                    RelationType::class,
                    array(
                        'label' => 'form.label.edit.site_user.groups',
                        'class' => Model\SiteGroup::class,
                        'multiple' => true,
                        'required' => false,
                    ),
                ),
                array(
                    'PlainPassword',
                    BaseType\RepeatedType::class,
                    array(
                        'type' => 'password',
                        'first_options'  => array(
                            'label' => 'form.label.edit.site_user.password',
                            'required' => false,
                        ),
                        'second_options' => array(
                            'label' => 'form.label.edit.site_user.repeat_password',
                            'required' => false,
                        ),
                        'required' => false,
                    ),
                ),
                array(
                    'DefaultRoles',
                    BaseType\ChoiceType::class,
                    array(
                        'label' => 'form.label.edit.site_user.roles',
                        'multiple' => true,
                        'expanded' => true,
                        'choices' => $roleTypeEvent->getTypeRoles(),
                        'attr' => array(
                            'trans-hint' => true,
                        ),
                    ),
                ),
/*
                array(
                    'Photos',
                    BaseType\CollectionType::class,
                    array(
                        'entry_type' => PhotoUploadType::class,
                        'entry_options'  => array(
                            'data_class' => Photo::class,
                            'required' => false,
                            'constraints' => array(
                                new Image(array(
                                    //'minHeight' => 10,
                                    //'maxHeight' => 100,
                                )),
                            ),
                        ),
                        'allow_add' => true,
                        'allow_delete'  => true,
                    ),
                ),
*/
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'backend_base_backend_siteuser_index',
                    ),
                ),
            ),
        ));
    }
}