<?php
namespace Backend\BaseBundle\EventListener\ControllerConfig;

use Backend\BaseBundle\Form\Type\RelationType;
use Backend\BaseBundle\Form\Type\SubmitBackType;
use JMS\DiExtraBundle\Annotation\Service;
use JMS\DiExtraBundle\Annotation\Tag;
use JMS\DiExtraBundle\Annotation\Inject;
use JMS\DiExtraBundle\Annotation\InjectParams;
use Backend\BaseBundle\Event\Controller;
use Symfony\Component\EventDispatcher\EventDispatcherInterface;
use Backend\BaseBundle\Model;
use Symfony\Component\Routing\Router;
use Symfony\Component\Form\Extension\Core\Type as BaseType;
use Symfony\Component\Validator\Constraints\NotBlank;

/**
 * @Service
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\ConfigEvent::EVENT_CONFIG, "method": "onConfig"})
 * @Tag("kernel.event_listener", attributes = {"public": false, "event": Controller\MenuEvent::EVENT_MENU, "method": "onMenu"})
 */
class SiteGroupListener
{
    const CONTROLLER_NAME = 'BackendBaseBundle:Backend/SiteGroup';

    /** @var  EventDispatcherInterface */
    protected $eventDispatcher;

    /** @var  Router */
    protected $router;

    /**
     * @InjectParams()
     */
    public function __construct(EventDispatcherInterface $eventDispatcher, Router $router)
    {
        $this->eventDispatcher = $eventDispatcher;
        $this->router = $router;
    }

    public function onMenu(Controller\MenuEvent $event)
    {
        $isActive = preg_match('/^backend_base_backend_sitegroup_/i', $event->getRequestRouteName());
        $user = $event->getUser();
        if(!$user->isSuperAdmin()){
            return;
        }
        $slug = $user->getSiteSlug();
        $event->addMenu(self::CONTROLLER_NAME, array(
            'menu.site_user' => array(
                'class' => 'fa fa-user',
                'active' => $isActive,
                'target' => array(
                    'menu.site_group' => array(
                        'class' => 'fa fa-users',
                        'active' => $isActive,
                        'target' => $this->router->generate('backend_base_backend_sitegroup_index', array('slug' => $slug)),
                    ),
                ),
            ),
        ));

    }

    public function onConfig(Controller\ConfigEvent $event)
    {
        if($event->getControllerName() != self::CONTROLLER_NAME){
            return;
        }
        $event->stopPropagation();

        $roleTypeEvent = new Controller\TypeRolesEvent();
        $this->eventDispatcher->dispatch($roleTypeEvent::EVENT_TYPE_ROLES, $roleTypeEvent);
        $event->setConfig(array(
            'route' => array(
                'sort' => true,
                'index' => true,
                'edit' => true,
                'new' => true,
                'delete' => true,
            ),
            'index' => array(
                '＃' => 'Id',
                'index.site_group.name' => 'Name',
                'index.site_group.created_at' => 'CreatedAt',
            ),
            'security' => array(
                'new' =>  'ROLE_SUPERADMIN',
                'edit' => 'ROLE_SUPERADMIN',
                'delete' => 'ROLE_SUPERADMIN',
            ),
            'batch' => array(
                'batch.site_group.delete' => array(
                    'url' => 'backend_base_backend_sitegroup_batchdelete',
                    'security' => 'ROLE_SUPERADMIN',
                ),
            ),
            'action' => array(
                'action.edit' => array(
                    'class' => 'fa fa-pencil-square-o',
                    'action' => 'edit'
                ),
                'action.delete' => array(
                    'class' => 'fa fa-trash',
                    'action' => 'delete',
                    'confirm' => 'message.confirm.delete',
                ),
            ),
            'form' => array(
                array(
                    'Id',
                    BaseType\HiddenType::class,
                    array(),
                ),
                array(
                    'Name',
                    BaseType\TextType::class,
                    array(
                        'label' => 'form.label.edit.site_group.name',
                        'required' => true,
                        'constraints' => array(
                            new NotBlank(),
                        ),
                    ),
                ),
                array(
                    'SiteUsers',
                    RelationType::class,
                    array(
                        'label' => 'form.label.edit.site_group.users',
                        'class' => Model\SiteUser::class,
                        'multiple' => true,
                        'required' => false,
                    ),
                ),
                array(
                    'DefaultRoles',
                    BaseType\ChoiceType::class,
                    array(
                        'required' => false,
                        'label' => 'form.label.edit.site_group.roles',
                        'multiple' => true,
                        'expanded' => true,
                        'choices' => $roleTypeEvent->getTypeRoles(),
                        'attr' => array(
                            'trans-hint' => true,
                        ),
                    ),
                ),
                array(
                    'submit',
                    SubmitBackType::class,
                    array(
                        'with_back' => true,
                        'with_reset' => true,
                        'back_route' => 'backend_base_backend_sitegroup_index',
                    ),
                ),
            ),
        ));
    }
}