<?php
namespace Backend\BaseBundle\Command;

use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputArgument;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Output\OutputInterface;
use Backend\BaseBundle\Model;


class ChangePasswordCommand extends ContainerAwareCommand
{
    /**
     * @see Command
     */
    protected function configure()
    {
        $this
            ->setName('dgfactor:siteuser:change-password')
            ->setDescription('變更後台帳號密碼.')
            ->setDefinition(array(
                new InputArgument('slug', InputArgument::REQUIRED, '站台代碼'),
                new InputArgument('username', InputArgument::REQUIRED, '帳號'),
                new InputArgument('password', InputArgument::REQUIRED, '密碼'),
            ))
            ->setHelp(<<<EOT
The <info>fos:user:change-password</info> command changes the password of a user:

  <info>php app/console fos:user:change-password matthieu</info>

This interactive shell will first ask you for a password.

You can alternatively specify the password as a second argument:

  <info>php app/console fos:user:change-password matthieu mypassword</info>

EOT
            );
    }

    /**
     * @see Command
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        $username = $input->getArgument('username');
        $password = $input->getArgument('password');
        $slug   = $input->getArgument('slug');
        $site = Model\SiteQuery::create()->findOneBySlug($slug);
        if(!$site){
            $output->writeln(sprintf('找不到站台 <error>%s</error>', $slug));
            return;
        }
        $siteUserProvider = $this->getContainer()->get('site_user_provider');
        $siteUserProvider->setSlug($slug);
        $siteUserManager = $this->getContainer()->get('site_user_manager');

        $siteUser = $siteUserProvider->loadUserByUsernameOrEmail($username);

        if(!$siteUser){
            $output->writeln(sprintf('找不到帳號 <error>%s</error>', $username));
            return;
        }

        $siteUser->setPlainPassword($password);
        $siteUserManager->updateUser($siteUser);

        $output->writeln(sprintf('Changed password for user <comment>%s</comment>', $username));
    }

    /**
     * @see Command
     */
    protected function interact(InputInterface $input, OutputInterface $output)
    {
        if (!$input->getArgument('slug')) {
            $slug = $this->getHelper('dialog')->askAndValidate(
                $output,
                '站台代碼:',
                function($slug) {
                    if (empty($slug)) {
                        throw new \Exception('站台代碼不能為空');
                    }

                    return $slug;
                }
            );
            $input->setArgument('slug', $slug);
        }
        if (!$input->getArgument('username')) {
            $username = $this->getHelper('dialog')->askAndValidate(
                $output,
                'Please give the username:',
                function($username) {
                    if (empty($username)) {
                        throw new \Exception('Username can not be empty');
                    }

                    return $username;
                }
            );
            $input->setArgument('username', $username);
        }

        if (!$input->getArgument('password')) {
            $password = $this->getHelper('dialog')->askHiddenResponseAndValidate(
                $output,
                'Please enter the new password:',
                function($password) {
                    if (empty($password)) {
                        throw new \Exception('Password can not be empty');
                    }

                    return $password;
                }
            );
            $input->setArgument('password', $password);
        }
    }
}
