<?php

namespace Backend\BaseBundle\Command;

use Backend\BaseBundle\Service\LocalShell;
use Symfony\Bundle\FrameworkBundle\Command\ContainerAwareCommand;
use Symfony\Component\Console\Input\InputInterface;
use Symfony\Component\Console\Input\InputOption;
use Symfony\Component\Console\Output\OutputInterface;
use Symfony\Component\Filesystem\Filesystem;
use Symfony\Component\Finder\Finder;
use Symfony\Component\HttpFoundation\File\File;
use Symfony\Component\HttpKernel\Kernel;

class DocumentBuildCommand extends ContainerAwareCommand
{
    /**
     * @see Command
     */
    protected function configure()
    {
        $this
            ->setName('dgfactor:document:build')
            ->setDescription('讀取文件檔案並建立.')
            ->addOption('force', null, InputOption::VALUE_NONE, '請加入這個參數，確認執行')
            ->setHelp(<<<EOT
<info>dgfactor:document:build</info> 讀取文件檔案並建立.
EOT
            );
    }

    /**
     * @see Command
     */
    protected function execute(InputInterface $input, OutputInterface $output)
    {
        if(!$input->getOption('force')){
            $output->writeln("<error>請加上 --force 確認執行</error>");
            return;
        }

        $documentFolder = $this->createDocumentFolder();
        $this->makeSureConfigFile($documentFolder);
        $this->cleanDocumentSource($documentFolder, $output);
        $this->copyDocuments($documentFolder, $output);
        $this->buildDocument($documentFolder, $output);
    }

    protected function createDocumentFolder()
    {
        $fileSystem = new Filesystem();
        $container = $this->getContainer();
        /** @var Kernel $kernel */
        $kernel = $container->get('kernel');
        $folder = $kernel->getRootDir()."/../doc";
        $fileSystem->mkdir($folder);
        return $folder;
    }

    protected function copyDocuments($folder, OutputInterface $output)
    {
        $fileSystem = new Filesystem();
        /** @var File $file */
        foreach ($this->findDocuments($output) as $file) {
            $tmp = explode("/", $file->getPath());
            $bundleName = $tmp[count($tmp)-3];
            $target = $folder."/src/$bundleName.{$file->getFilename()}";
            if (file_exists($target)) {
                continue;
            }
            $fileSystem->copy($file->getPathname(), $target);
        }
    }

    /**
     * @return \Generator
     */
    protected function findDocuments(OutputInterface $output)
    {
        $documentSearchPath = array();
        $container = $this->getContainer();
        $kernel = $container->get('kernel');

        $baseDocumentPath = "{$kernel->getRootDir()}/Resources/doc";
        if(is_dir($baseDocumentPath)){
            $output->writeln("<info>search $baseDocumentPath</info>");
            $documentSearchPath[] = $baseDocumentPath;
        }

        foreach($kernel->getBundles() as $bundle){
            $path = "{$bundle->getPath()}/Resources/doc";
            if(is_dir($path)){
                $output->writeln("<info>search {$bundle->getName()}</info>");
                $documentSearchPath[] = $path;
            }
        }

        $finder = new Finder();
        foreach($finder->in($documentSearchPath)->files()->name('*.php') as $file){
            yield $file;
        }
    }

    protected function buildDocument($documentFolder, OutputInterface $output)
    {
        /** @var LocalShell $localShell */
        $localShell = $this->getContainer()->get('local_shell');
        $localShell->execute(
            'apidoc',
            $documentFolder,
            $output
        );
    }

    protected function makeSureConfigFile($documentFolder)
    {
        if (file_exists($filePath = $documentFolder."/apidoc.json")) {
            return;
        }
        $content = array(
            "name" => "project api",
            "version" => "0.0.1",
            "description" => "project API 文件",
            "title" => "",
            "url" => "",
        );
        file_put_contents(
            $filePath,
            json_encode($content, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE)
        );
    }

    protected function cleanDocumentSource($documentFolder, OutputInterface $output)
    {
        /** @var LocalShell $localShell */
        $localShell = $this->getContainer()->get('local_shell');
        $localShell->execute(
            'rm -rf src',
            $documentFolder,
            $output
        );
    }
}
